#!/usr/bin/env python3

import parglare as Parglare
from abc import ABC, abstractmethod

class Grammar(ABC):
    def __init__(self, grammar, keywords, recognizers=None, debug=False):
        assert isinstance(grammar, str)
        self.grammar = grammar
        self.keywords = keywords
        self.recognizers = recognizers
        self.debug = debug

    @abstractmethod
    def get_action(self,inp):
        """
        This method should return a dictionary of actions for the parser.
        The keys are the names of the actions and the values are the functions
        that implement the actions.
        """
        pass
    
    def interpret_result(self, result):
        # Do nothing
        return result

    def _get_parser(self, actions={}):
        if self.recognizers is not None:
            g = Parglare.Grammar.from_string(self.grammar, recognizers=self.recognizers, debug=self.debug)
        else:
            g = Parglare.Grammar.from_string(self.grammar)
        parser = Parglare.Parser(g, debug=self.debug, actions=actions)
        return parser

    def compile(self, code):
        parser = self._get_parser()
        result = parser.parse(code)
        return result

    def run(self, code, inp):
        actions = self.get_action(inp)
        parser = self._get_parser(actions)
        result = parser.parse(code)
        return self.interpret_result(result)