from frame.tools.z3_dsl import Z3Program

def _format_args(args):
    """Return a comma-separated string like 'x_0=a, x_1=b, …' for the given arguments."""
    return ", ".join(f"x_{i}={arg}" for i, arg in enumerate(args))


class Z3Template:
    """A template for Z3 programs."""
    def __init__(self, code: str, *args):
        self.code = code
        self.program = Z3Program.from_code(code)
        self.args = args

    def __str__(self):
        return self.code

    def set_args(self, *args):
        """Set the arguments for the Z3 program."""
        self.args = args

    def run(self):
        """Run the Z3 program with the given arguments. Can only be run on predicates."""
        code = f"""
            params 0;
            bounded params 0;
            p_0 := Pred(
            {self.program.dsl()}
            );
            ReturnExpr None;
            ReturnPred p_0({_format_args([str(arg.to_z3()) for arg in self.args])});
        """
        return Z3Program.from_code(code).run()

    def check_example(self, is_function: bool, example: tuple, timeout=1):
        """Check if the Z3 program is true for the given example."""
        
        if is_function:
            # assuming the last argument is the output
            # Note(_; 5/4): and assuming the example is a tuple of natural numbers
            code = f"""
                params 0;
                bounded params 0;
                f_0 := Pred(
                {self.program.dsl()}
                );
                ReturnExpr None;
                ReturnPred f_0({_format_args([str(arg.to_z3()) for arg in example])}) == {example[-1].to_z3()};
            """
        else:
            code = f"""
                params 0;
                bounded params 0;
                p_0 := Pred(
                {self.program.dsl()}
                );
                ReturnExpr None;
                ReturnPred p_0({_format_args([str(arg.to_z3()) for arg in example])});
            """
        return Z3Program.from_code(code).run(timeout=timeout)
    