import pytest

from frame.productions.concepts.constant import ConstantRule
from frame.knowledge_base.entities import (Example, ExampleStructure, ExampleType, Concept, ConceptType)

# ==============================
# Fixtures
# ==============================

@pytest.fixture
def constant_rule():
    """Create a fresh NegateRule instance for each test."""
    return ConstantRule()

# ==============================
# Basic Rule Tests
# ==============================

class TestConstantRuleBasics:
    """Test basic functionality of the ConstantRule class."""

    def test_get_input_types(self, constant_rule):
        """Test that get_input_types returns the correct types."""
        input_types = constant_rule.get_input_types()
        assert isinstance(input_types, list), "Should return a list"
        assert len(input_types) > 0, "Should return at least one input type combination"

        # Each input type should be a list 
        for type_combo in input_types:
            assert isinstance(type_combo, tuple), "Each input type handled should be a tuple"
            assert len(type_combo) == 2, "Each type spec should have 2 elements"

    def test_get_valid_parameterizations(self, constant_rule, square, two_concept):
        """Test that get_valid_parameterizations returns valid parameterizations."""
        # Test with example-based case
        params_function_example = constant_rule.get_valid_parameterizations(square)
        assert isinstance(params_function_example, list), "Should return a list"
        num_square_examples = len(square.examples.get_examples())
        assert (
            len(params_function_example) == num_square_examples
            ), "Should have one parameterization per example" 

    def test_can_apply_function_example(self, constant_rule, square):
        """Test that can_apply returns True for valid constant + function input."""
        square_example = list(square.examples.get_examples())[0]
        assert constant_rule.can_apply(
            square, example=square_example
        ), "Should allow application of constant rule on function"

# ================================
# Constant Rule Application Tests
# ================================
        
class TestConstantRuleApplication:
    """Test application of constant rules."""

    def test_create_from_predicate_example(self, constant_rule, is_even):
        """Test creating constant from predicate example"""
        # Get example and verify its value
        even_example = sorted(list(is_even.examples.get_examples()))[-1]
        assert even_example.value == (18,), "Example from fixture should be 18"

        # Create constant
        eighteen = constant_rule.apply(is_even, example=even_example)

        # Test concept name
        assert eighteen.name == "constant_(18_from_is_even)"

        # Test computation
        eighteen_value = eighteen.compute()
        assert eighteen_value == 18, """Value of 18 is 18"""
        assert (
            list(eighteen.examples.get_examples())[0].value == (18,)
        ), """18 should be in examples for eighteen concept"""
        assert len(eighteen.examples.get_examples()) == 1, """Constant should have exactly one example"""        

    def test_create_from_function_example(self, constant_rule, square):
        """Test creating constant from function example"""

        # Get example and verify its value
        square_example = sorted(list(square.examples.get_examples()))[-1]
        assert square_example.value == (19, 361), "Last example should be square of 19"

        # Create constant
        square_const = constant_rule.apply(square, example=square_example)

        # Test concept name
        assert square_const.name == "constant_(361_from_square)"

        # Test computation
        square_const_value = square_const.compute()
        assert square_const_value == 361, """Square of 19 is 361"""
        assert (
            list(square_const.examples.get_examples())[0].value == (361,)
        ), """0 should be in examples for square_const concept"""
        assert len(square_const.examples.get_examples()) == 1, """Constant should have exactly one example"""

# ==============================
# Error Case Tests
# ==============================
        
class TestConstantRuleErrors:
    """Test constant rule cases for the ConstantRule class."""

    def test_cannot_apply_with_wrong_number_of_inputs(self, constant_rule, square, three_concept):
        """Test that can_apply returns False when wrong number of inputs provided."""
        # Test with too many inputs
        assert not constant_rule.can_apply(three_concept, square), "Should not allow 2 inputs"
        # Test with too few inputs
        assert not constant_rule.can_apply(), "Should not allow 0 inputs"
        assert not constant_rule.can_apply(three_concept), "Should not allow 1 input without example"

    def test_cannot_apply_with_non_concept_inputs(self, constant_rule, square, three_concept):
        """Test that can_apply returns False when non-concept inputs provided."""
        # Test with non-concept inputs
        assert not constant_rule.can_apply("not a concept", square), "Should not allow string input"
        assert not constant_rule.can_apply(three_concept, "not a concept"), "Should not allow string input"

    def test_cannot_apply_with_wrong_concept_types(self, constant_rule, add_multiply, three_concept):
        """Test that can_apply returns False when wrong concept types provided."""
        assert not constant_rule.can_apply(three_concept), "Should not allow constant input"
        assert not constant_rule.can_apply(add_multiply), "Should not allow function output length > 1"

    def test_cannot_apply_with_invalid_example(self, constant_rule, square):
        """Test that can_apply returns False when invalid example provided."""
        # Test with invalid example value
        invalid_example = Example(value=(1,1,2,3), example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,ExampleType.NUMERIC,ExampleType.NUMERIC,
                             ExampleType.NUMERIC),
            input_arity=4,)
        )
        assert not constant_rule.can_apply(square, example=invalid_example), "Should not allow invalid example"