#!/usr/bin/env python3
"""
Script to download the missing model logos with alternative URLs.
"""
import requests
import sys
from pathlib import Path

def download_logo(url, filename, description):
    """Download a logo from URL and save it with the given filename."""
    try:
        print(f"📥 Downloading {description}...")
        headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
        }
        response = requests.get(url, stream=True, timeout=30, headers=headers)
        response.raise_for_status()
        
        logo_path = Path("../results/logos") / filename
        with open(logo_path, 'wb') as f:
            for chunk in response.iter_content(chunk_size=8192):
                f.write(chunk)
        
        print(f"✅ Saved {description} to {logo_path}")
        return True
        
    except Exception as e:
        print(f"❌ Failed to download {description}: {e}")
        return False

def main():
    print("🔽 Downloading missing AI model logos with alternative URLs...")
    
    # Alternative URLs for the missing logos
    missing_logos = [
        {
            "url": "https://upload.wikimedia.org/wikipedia/commons/thumb/0/04/ChatGPT_logo.svg/1024px-ChatGPT_logo.svg.png",
            "filename": "openai.png",
            "description": "OpenAI logo (alternative)"
        },
        {
            "url": "https://pbs.twimg.com/profile_images/1692935552478310400/E8CC5uxp_400x400.jpg",
            "filename": "anthropic.png", 
            "description": "Anthropic logo (alternative)"
        },
        {
            "url": "https://docs.mistral.ai/img/logo.svg",
            "filename": "mistral.svg",
            "description": "Mistral AI logo (SVG)"
        }
    ]
    
    successful_downloads = 0
    
    for logo_info in missing_logos:
        if download_logo(logo_info["url"], logo_info["filename"], logo_info["description"]):
            successful_downloads += 1
    
    print(f"\n📊 Additional Download Summary:")
    print(f"   ✅ Successfully downloaded: {successful_downloads}/{len(missing_logos)} missing logos")
    
    # Check total logos now available
    logos_dir = Path("../results/logos")
    available_logos = list(logos_dir.glob("*"))
    print(f"   📁 Total logos available: {len(available_logos)}")
    for logo in sorted(available_logos):
        print(f"      • {logo.name}")

if __name__ == "__main__":
    main()