#!/usr/bin/env python3
"""
Script to download model logos from the internet.
"""
import requests
import sys
from pathlib import Path

def download_logo(url, filename, description):
    """Download a logo from URL and save it with the given filename."""
    try:
        print(f"📥 Downloading {description}...")
        response = requests.get(url, stream=True, timeout=30)
        response.raise_for_status()
        
        logo_path = Path("../results/logos") / filename
        with open(logo_path, 'wb') as f:
            for chunk in response.iter_content(chunk_size=8192):
                f.write(chunk)
        
        print(f"✅ Saved {description} to {logo_path}")
        return True
        
    except Exception as e:
        print(f"❌ Failed to download {description}: {e}")
        return False

def main():
    # Create logos directory
    logos_dir = Path("../results/logos")
    logos_dir.mkdir(parents=True, exist_ok=True)
    
    print("🔽 Downloading AI model logos...")
    
    # Model logo URLs (these are publicly available logos)
    logos_to_download = [
        {
            "url": "https://cdn.openai.com/API/logo-assets/openai-logomark.png",
            "filename": "openai.png",
            "description": "OpenAI logo (GPT models)"
        },
        {
            "url": "https://www.anthropic.com/images/logos/claude-logo.png", 
            "filename": "anthropic.png",
            "description": "Anthropic logo (Claude models)"
        },
        {
            "url": "https://avatars.githubusercontent.com/u/115825241?s=200&v=4",
            "filename": "deepseek.png", 
            "description": "DeepSeek logo"
        },
        {
            "url": "https://avatars.githubusercontent.com/u/47204120?s=200&v=4",
            "filename": "qwen.png",
            "description": "Qwen/Alibaba logo"
        },
        {
            "url": "https://avatars.githubusercontent.com/u/60539444?s=200&v=4",
            "filename": "zhipu.png",
            "description": "Zhipu AI logo (GLM models)"
        },
        {
            "url": "https://pbs.twimg.com/profile_images/1683899100922511378/5lY42eHs_400x400.jpg",
            "filename": "x.png",
            "description": "X logo (Grok models)"
        },
        {
            "url": "https://avatars.githubusercontent.com/u/54048788?s=200&v=4",
            "filename": "meta.png",
            "description": "Meta logo (Llama models)"
        },
        {
            "url": "https://mistral.ai/images/logo_hubc88c4ece272ce9367744af5507ca0dab_1117_256x0_resize_q90_h2_lanczos_3.webp",
            "filename": "mistral.png", 
            "description": "Mistral AI logo"
        },
        {
            "url": "https://avatars.githubusercontent.com/u/2810941?s=200&v=4",
            "filename": "google.png",
            "description": "Google logo (Gemini models)"
        }
    ]
    
    successful_downloads = 0
    total_downloads = len(logos_to_download)
    
    for logo_info in logos_to_download:
        if download_logo(logo_info["url"], logo_info["filename"], logo_info["description"]):
            successful_downloads += 1
    
    print(f"\n📊 Download Summary:")
    print(f"   ✅ Successfully downloaded: {successful_downloads}/{total_downloads} logos")
    
    if successful_downloads < total_downloads:
        print(f"   ⚠️ Failed downloads: {total_downloads - successful_downloads}")
        print("   💡 You can manually add missing logos to ../results/logos/")
    
    print(f"\n📁 Logos saved to: {logos_dir.absolute()}")

if __name__ == "__main__":
    main()