#!/usr/bin/env python3
"""
Script to create accuracy and F1 score plots for both exact model and binary self tasks.
"""
import sys
from pathlib import Path
sys.path.append('../src')

from visualizer import ResultsVisualizer

def main():
    # Initialize visualizer
    visualizer = ResultsVisualizer()
    
    # Define prediction files to analyze
    exact_model_file = "../results/predictions/predictions_exact_model.jsonl"
    binary_self_file = "../results/predictions/predictions_binary_self.jsonl"
    
    # Create plots directory
    plots_dir = Path("../results/plots")
    plots_dir.mkdir(parents=True, exist_ok=True)
    
    print("📊 Creating accuracy and F1 score plots...")
    
    # Exact Model Task Plots
    print("\n🔍 Exact Model Task:")
    
    print("  Creating exact model accuracy plot...")
    fig1 = visualizer.create_exact_model_accuracy_plot(
        exact_model_file, 
        save_path="../results/plots/exact_model_accuracy.pdf"
    )
    
    print("  Creating exact model F1 score plot...")
    fig2 = visualizer.create_exact_model_f1_plot(
        exact_model_file, 
        save_path="../results/plots/exact_model_f1.pdf"
    )
    
    # Binary Self Task Plots
    print("\n🔵 Binary Self Task:")
    
    print("  Creating binary model accuracy plot...")
    fig3 = visualizer.create_binary_model_accuracy_plot(
        binary_self_file, 
        save_path="../results/plots/binary_model_accuracy.pdf"
    )
    
    print("  Creating binary model F1 score plot...")
    fig4 = visualizer.create_binary_model_f1_plot(
        binary_self_file, 
        save_path="../results/plots/binary_model_f1.pdf"
    )
    
    # Report results
    plots_created = sum(1 for fig in [fig1, fig2, fig3, fig4] if fig is not None)
    
    if plots_created == 4:
        print("\n✅ All accuracy and F1 plots created successfully!")
        print("\n📄 Generated plots:")
        print("   🔍 Exact Model Task:")
        print("      • Accuracy: ../results/plots/exact_model_accuracy.pdf")
        print("      • F1 Score: ../results/plots/exact_model_f1.pdf")
        print("   🔵 Binary Self Task:")
        print("      • Accuracy: ../results/plots/binary_model_accuracy.pdf")
        print("      • F1 Score: ../results/plots/binary_model_f1.pdf")
    elif plots_created > 0:
        print(f"\n⚠️ {plots_created}/4 plots created successfully")
        if fig1 is not None:
            print("   ✅ Exact model accuracy created")
        if fig2 is not None:
            print("   ✅ Exact model F1 created")
        if fig3 is not None:
            print("   ✅ Binary model accuracy created")  
        if fig4 is not None:
            print("   ✅ Binary model F1 created")
    else:
        print("\n❌ Failed to create plots - no data available")
    
    print(f"\n💡 Plot details:")
    print("   • Exact Model F1: Macro-averaged across all target models")
    print("   • Binary Model F1: Standard binary classification F1 score")
    print("   • Large fonts and clean percentage labels for publication quality")

if __name__ == "__main__":
    main()