import sys
import argparse
import yaml
import base64
from pathlib import Path

from browser_env import ScriptBrowserEnv
from webagents_step.utils.data_prep import DotDict

def main():
    parser = argparse.ArgumentParser(description='Observe environment for a given URL using config.')
    parser.add_argument('--config', type=str, required=True, help='Path to YAML config file')
    parser.add_argument('--url', type=str, required=True, help='URL to observe')
    args = parser.parse_args()

    # Load config
    with open(args.config, 'r') as f:
        config = yaml.safe_load(f)
    config = DotDict(config)

    # Prepare a temp config file with start_url injected
    import tempfile
    import json
    temp_config = dict(config.to_dict())
    temp_config['start_url'] = args.url
    with tempfile.NamedTemporaryFile('w', delete=False, suffix='.json') as tf:
        json.dump(temp_config, tf)
        temp_config_path = tf.name

    # Instantiate environment
    env = ScriptBrowserEnv(
        headless=getattr(config.env, 'headless', True),
        observation_type='html',
        current_viewport_only=False,
        viewport_size={"width": 1280, "height": 720},
        global_config=config
    )
    obs, info = env.reset(options={'config_file': temp_config_path})

    print('--- Environment Observation (text) ---')
    print(obs['text'])
    print('\n--- Environment Observation (image, base64) ---')
    print(obs['image'])
    env.close()

if __name__ == '__main__':
    main() 