from typing import Dict, Generic, TypeVar

import optax
from chex import PRNGKey
from flashbax.buffers.trajectory_buffer import TrajectoryBufferState
from flax.core.scope import FrozenVariableDict
from jax import Array
from jumanji.env import State
from typing_extensions import NamedTuple, TypeAlias

from mava.types import Observation

Metrics = Dict[str, Array]


class Transition(NamedTuple):
    """Transition for recurrent Q-learning."""

    obs: Observation
    action: Array
    reward: Array
    terminal: Array
    term_or_trunc: Array
    # Even though we use a trajectory buffer we need to store both obs and next_obs.
    # This is because of how the `AutoResetWrapper` returns obs at the end of an episode.
    next_obs: Observation


BufferState: TypeAlias = TrajectoryBufferState[Transition]


class QNetParams(NamedTuple):
    """Double Q-learning network parameters."""

    online: FrozenVariableDict
    target: FrozenVariableDict


class ActionSelectionState(NamedTuple):
    """Everything used for action selection apart from the observation."""

    online_params: FrozenVariableDict
    hidden_state: Array
    time_steps: int
    key: PRNGKey


class ActionState(NamedTuple):
    """The carry in the interaction loop."""

    action_selection_state: ActionSelectionState
    env_state: State
    buffer_state: BufferState
    obs: Observation
    terminal: Array
    term_or_trunc: Array


class QMIXParams(NamedTuple):
    online: FrozenVariableDict
    target: FrozenVariableDict
    mixer_online: FrozenVariableDict
    mixer_target: FrozenVariableDict


QLearningParams = TypeVar("QLearningParams", QNetParams, QMIXParams)


class LearnerState(NamedTuple, Generic[QLearningParams]):
    """State of the learner in an interaction-training loop."""

    # Interaction vars
    obs: Observation
    terminal: Array
    term_or_trunc: Array
    hidden_state: Array
    env_state: State
    time_steps: Array

    # Train vars
    train_steps: Array
    opt_state: optax.OptState

    # Shared vars
    buffer_state: TrajectoryBufferState
    params: QLearningParams
    key: PRNGKey


class TrainState(NamedTuple, Generic[QLearningParams]):
    """The carry in the training loop."""

    buffer_state: BufferState
    params: QLearningParams
    opt_state: optax.OptState
    train_steps: Array
    key: PRNGKey
