import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.vdn import VDNMixer
from modules.mixers.qmix import QMixer
from utils.rl_utils import build_td_lambda_targets
import torch as th
from torch.optim import RMSprop, Adam
import numpy as np
from modules.opponent.contrastive_encoder import ProtoQueue, info_nce_loss

import pdb

class QLearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger
        
        self.params = list(mac.parameters())

        self.last_target_update_episode = 0
        self.device = th.device('cuda' if args.use_cuda  else 'cpu')

        self.mixer = None
        if args.mixer is not None:
            if args.mixer == "vdn":
                self.mixer = VDNMixer()
            elif args.mixer == "qmix":
                self.mixer = QMixer(args)
            else:
                raise ValueError("Mixer {} not recognised.".format(args.mixer))
            self.params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)



        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        self.log_stats_t = -self.args.learner_log_interval - 1
        
        self.train_t = 0

        self.use_contrastive = getattr(args, 'use_contrastive', False)
        if self.use_contrastive:
            self.z_dim = getattr(args, 'z_dim', 16)
            self.n_neg = getattr(args, 'nce_negatives', 64)
            self.temp = getattr(args, 'nce_temperature', 0.07)
            self.proto_queue = ProtoQueue(self.z_dim, max_size=getattr(args, 'nce_queue_size', 4096), momentum=getattr(args, 'nce_momentum', 0.9), device=self.device)
            self.nce_loss_weight = getattr(args, 'nce_loss_weight', 1.0)

        self.encoder_params = list(mac.encoder_params())
        #self.encoder_optimizer = Adam(params=self.encoder_params, lr=float(args.encoder_lr))
        
        self.ctx_window = getattr(args, 'ctx_window', 8)
        self.z_dim = getattr(args, 'z_dim', 16)
        self.ctx_hid = getattr(args, 'ctx_hid', 64)

        self.ctx_builder = mac.ctx_builder
        self.ctx_encoder = mac.ctx_encoder
        self.target_ctx_encoder = copy.deepcopy(self.ctx_encoder)

        self.gamma = args.gamma
        self.grad_norm_clip = getattr(args, 'grad_norm_clip', 10)
        self.nce_loss_weight_init = self.nce_loss_weight
        self.nce_loss_weight_final = getattr(args, 'nce_loss_weight_final', 0.0)
        self.nce_loss_decay_steps = getattr(args, 'nce_loss_decay_steps', int(5e6))

        if self.args.optimizer == 'adam':
            self.optimiser = Adam(params=self.params + list(self.encoder_params),  lr=args.lr)
        else:
            self.optimiser = RMSprop(params=self.params, lr= float(args.lr), alpha=args.optim_alpha, eps=args.optim_eps)

    def get_current_nce_weight(self, t_env):
        frac = min(t_env/ float(self.nce_loss_decay_steps), 1.0)
        return self.nce_loss_weight_init * (1 - frac) + self.nce_loss_weight_final * frac

    def set_contrastive(self, con):
        self.use_contrastive = con


    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int):
        # Get the relevant quantities
        rewards = batch["reward"][:, :-1]
        actions = batch["actions"][:, :-1]
        terminated = batch["terminated"][:, :-1].float()
        mask = batch["filled"][:, :-1].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]
        
        B = batch.batch_size
        T = batch.max_seq_length

        # Push episode prototype(s) into queue
        '''
        if self.use_contrastive and 'z_proto' in batch.data.transition_data:
            ep_proto = batch['z_proto'][:, 0, :].to(self.proto_queue.protos.device)  # representative
            proto_idxs, pos_proto = self.proto_queue.push(ep_proto)
        else:
            proto_idxs = None
            pos_proto = ep_proto
        '''


        # Calculate estimated Q-Values
        mac_out = []
        z_ts = []
        self.mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            ctx_seq = self.ctx_builder.build_window(batch, t, self.ctx_window)  # [B, Tctx, C]
            z_seq = self.ctx_encoder(ctx_seq)  # [B, Tctx, Z]
            #z_t = z_seq[:, -1]
            z_ts.append(z_seq)
            #z_t = batch['z_t'][:, t, :] if ('z_t' in batch.data.transition_data and self.use_contrastive) else None
            agent_outs, _ = self.mac.forward(batch, t=t, z_t=z_seq)
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time
        z_ts = th.stack(z_ts, dim=1)



        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim
        chosen_action_qvals_back = chosen_action_qvals
        
        # Calculate the Q-Values necessary for the target
        target_mac_out = []
        target_z_ts = []
        self.target_mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            ctx_seq = self.ctx_builder.build_window(batch, t, self.ctx_window)  # [B, Tctx, C]
            z_seq = self.target_ctx_encoder(ctx_seq)  # [B, Tctx, Z]
            #z_t = z_seq[:, -1]
            target_z_ts.append(z_seq)
            #z_t = batch['z_t'][:, t, :] if ('z_t' in batch.data.transition_data and self.use_contrastive) else None
            target_agent_outs, _ = self.target_mac.forward(batch, t=t, z_t=z_seq)
            target_mac_out.append(target_agent_outs)

        # We don't need the first timesteps Q-Value estimate for calculating targets
        target_mac_out = th.stack(target_mac_out, dim=1)  # Concat across time
        target_z_ts = th.stack(target_z_ts, dim=1)

        # Mask out unavailable actions
        target_mac_out[avail_actions == 0] = -9999999

        # Max over target Q-Values
        if self.args.double_q:
            # Get actions that maximise live Q (for double q-learning)
            mac_out_detach = mac_out.clone().detach()
            mac_out_detach[avail_actions == 0] = -9999999
            cur_max_actions = mac_out_detach.max(dim=3, keepdim=True)[1]
            target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)

        else:
            target_max_qvals = target_mac_out.max(dim=3)[0]

        # Mix
        if self.mixer is not None:
            #z_seq = batch['z_t'][:, :-1, :] if ('z_t' in batch.data.transition_data and self.use_contrastive) else None
            #z_seq_tgt = batch['z_t'][:, 1:, :] if ('z_t' in batch.data.transition_data and self.use_contrastive) else None
            z_seq = z_ts[:, :-1, :].mean(dim=2)
            z_seq_tgt = target_z_ts.mean(dim=2)
            chosen_action_qvals = self.mixer(chosen_action_qvals, batch["state"][:, :-1], z_t=z_seq)
            target_max_qvals = self.target_mixer(target_max_qvals, batch["state"], z_t=z_seq_tgt)

        # Calculate 1-step Q-Learning targets
        #targets = rewards + self.args.gamma * (1 - terminated) * target_max_qvals
        targets = build_td_lambda_targets(rewards, terminated, mask, target_max_qvals, self.args.n_agents, self.args.gamma, self.args.td_lambda)
        # Td-error
        td_error = (chosen_action_qvals - targets.detach())

        mask = mask.expand_as(td_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask

        # Normal L2 loss, take mean over actual data
        loss_td = 0.5 * (masked_td_error ** 2).sum() / mask.sum()
        

        if self.use_contrastive:
            #z0 = batch['z_t'][:, 0, :].to(self.proto_queue.protos.device)  # [B, Z]

            z0 = (z_ts.mean(dim=2)[:,:-1] * mask).sum(dim=1) / mask.sum(dim=1)  # mean pool over time  positive

            proto_idxs, _ = self.proto_queue.push(z0)
            choice = th.randint(5, z_ts.size(1), (B,))
            anchors = z_ts.mean(dim=2)[th.arange(B), choice]

            pos_proto = z0.to('cuda')       # [B, Z]
            negs = self.proto_queue.sample_negatives(proto_idxs, self.n_neg)  # [B, n_neg, Z] or None
            nce_loss = info_nce_loss(anchors, pos_proto, negs, temperature=self.temp)

            current_weight = self.get_current_nce_weight(t_env)

            loss = loss_td + current_weight * nce_loss
            #loss = loss_td + self.nce_loss_weight * nce_loss
        else:
            nce_loss = th.tensor(0)
            loss = loss_td


        # Optimise
        self.optimiser.zero_grad()
        #self.encoder_optimizer.zero_grad()
        
        loss.backward()
        th.nn.utils.clip_grad_norm_(self.params, self.args.grad_norm_clip)
        #th.nn.utils.clip_grad_norm_(self.encoder_params, 5)
        self.optimiser.step()
        #self.encoder_optimizer.step()


        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("loss_td", loss_td.item(), t_env)
            self.logger.log_stat("loss_nce", nce_loss.item(), t_env)
            #self.logger.log_stat("grad_norm", grad_norm, t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat('before_mixer', (chosen_action_qvals_back * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("q_taken_mean", (chosen_action_qvals * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.log_stats_t = t_env
            


    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        self.target_ctx_encoder.load_state_dict(self.ctx_encoder.state_dict())
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()

    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        self.optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))
