import torch as th
import torch.nn as nn
import torch.nn.functional as F

class LISOAgent(nn.Module):
    """
    Agent network: takes in obs, strategy embedding, and outputs Q-values.
    """
    def __init__(self, input_shape, args):
        super(LISOAgent, self).__init__()
        self.args = args
        self.hidden_dim = args.rnn_hidden_dim
        self.n_actions = args.n_actions
        self.strategy_dim = args.strategy_embed_dim

        # Input includes obs + strategy embedding
        self.fc1 = nn.Linear(input_shape + self.strategy_dim, args.hidden_dim)
        self.rnn = nn.GRUCell(args.hidden_dim, args.rnn_hidden_dim)
        self.fc2 = nn.Linear(args.rnn_hidden_dim, self.n_actions)

    def init_hidden(self):
        # [batch, hidden_dim]
        return self.fc1.weight.new(1, self.hidden_dim).zero_()

    def forward(self, obs, strategy_emb, hidden_state):
        # obs: [batch, obs_dim]
        # strategy_emb: [batch, strategy_dim]
        x = th.cat([obs, strategy_emb], dim=-1)
        x = F.relu(self.fc1(x))
        h_in = hidden_state.reshape(-1, self.hidden_dim)
        h = self.rnn(x, h_in)
        q = self.fc2(h)
        return q, h
