#!/bin/zsh

# Set up directories
EXPERIMENT_DIR="experiment_results"
DETAILED_LOGS_DIR="$EXPERIMENT_DIR/detailed_logs"
SUMMARY_LOGS_DIR="$EXPERIMENT_DIR/summary_logs"
VISUALIZATION_DIR="$EXPERIMENT_DIR/visualizations"

# Create necessary directories
mkdir -p $DETAILED_LOGS_DIR
mkdir -p $SUMMARY_LOGS_DIR
mkdir -p $VISUALIZATION_DIR

# Define datasets and parameters
DATASETS=("elec" "aram")
STRATEGY="random_switch"
NUM_SEEDS=2
START_SEED=0

# Function to run a single experiment
run_experiment() {
    local dataset=$1
    local seed=$2
    
    echo "\n=== Running experiment for $dataset (seed $seed) ==="
    python run_experiment.py \
        --dataset $dataset \
        --splitting_strategy $STRATEGY \
        --num_seeds 1 \
        --start_seed $seed \
        --log_dir $DETAILED_LOGS_DIR \
        --results_dir $EXPERIMENT_DIR
}

# Function to generate visualizations for a dataset
generate_visualizations() {
    local dataset=$1
    
    echo "\n=== Generating visualizations for $dataset (seed 0) ==="
    python visualize_detailed_logs.py \
        --dataset $dataset \
        --strategy $STRATEGY \
        --seed 0 \
        --detailed_logs_dir $DETAILED_LOGS_DIR \
        --output_dir $VISUALIZATION_DIR
}

# Main execution
echo "Starting experiments for all datasets..."

# Run experiments for each dataset
for dataset in $DATASETS; do
    echo "\n===== Processing dataset: $dataset ====="
    
    # Run experiments for all seeds
    for ((seed=START_SEED; seed<START_SEED+NUM_SEEDS; seed++)); do
        run_experiment $dataset $seed
    done
    
    # Generate visualizations for seed 0
    generate_visualizations $dataset
done

# Compute aggregate statistics
echo "\n=== Computing aggregate statistics ==="
python aggregate_summary_logs.py

echo "\n=== All experiments completed ==="
echo "Results are saved in:"
echo "- Detailed logs: $DETAILED_LOGS_DIR"
echo "- Summary logs: $SUMMARY_LOGS_DIR"
echo "- Visualizations: $VISUALIZATION_DIR"
echo "- Aggregated statistics: $SUMMARY_LOGS_DIR/aggregated_summary_stats.csv" 