import json 
import logging
from collections import defaultdict 
from typing import Dict, Any

import numpy as np 
import pandas as pd

log = logging.getLogger(__name__)

def convert_defaultdict(d):
    if isinstance(d, defaultdict):
        d = dict(d)  # Convert the defaultdict to a normal dict
    for key, value in d.items():
        if isinstance(value, dict):
            d[key] = convert_defaultdict(value)  # Recursively convert nested dicts
    return d

def json_dump(to_dump, file_name): 
    with open(file_name, 'w') as f:
        log.info(f"Writing outputs to {file_name}")
        json.dump(to_dump, f, default=str)

def json_load(file_name: str) -> Dict[str, Any]:
    """
    Load a json file and return the contents as a dictionary.
    """
    with open(file_name, 'r') as f: 
        log.info(f"Reading outputs from {file_name}")
        results = json.load(f)
    return results

def get_ks(cfg): 
    assert cfg.kmin >= 0
    assert cfg.kmax >= cfg.kmin
    assert cfg.inc >= 0
    kpowers = np.arange(cfg.kmin, cfg.kmax + cfg.inc, step=cfg.inc)
    k_list = np.power(2, kpowers).astype(int)
    return k_list[::-1]

def pd_json_dump(df, file_name): 
    print(f"Writing dataframe to {file_name}")
    df.to_json(file_name)

def pd_json_load(file_name): 
    print(f"Reading dataframe from {file_name}")
    return pd.read_json(file_name)
    
