import numpy as np
import time
from .eoh_evolution import Evolution
import warnings
from joblib import Parallel, delayed
# from .evaluator_accelerate import add_numba_decorator # Temporarily commented out for Scheme C
import re
import concurrent.futures
import copy  # Import copy for deep copying dictionaries if needed
import traceback  # Ensure traceback is imported


class InterfaceEC():
    def __init__(self, pop_size, m, api_endpoint, api_key, llm_model, llm_use_local, llm_local_url, debug_mode, interface_prob, select, n_p, timeout, use_numba, **kwargs):

        # LLM settings
        self.pop_size = pop_size
        self.interface_eval = interface_prob
        # Ensure prompts object is correctly passed and is the modified PromptsCombined
        if not hasattr(interface_prob, 'prompts') or not hasattr(interface_prob.prompts, 'get_prompt'):
            raise ValueError(
                "The 'problem' instance must have a 'prompts' attribute with a 'get_prompt' method (likely modified PromptsCombined).")
        prompts = interface_prob.prompts

        # --- Call Evolution WITHOUT fixed_modules_code named argument ---
        self.evol = Evolution(
            api_endpoint,
            api_key,
            llm_model,
            llm_use_local,
            llm_local_url,
            debug_mode,
            prompts,  # prompts object now carries the fixed code info internally
            **kwargs  # Pass remaining kwargs
        )
        # -------------------------------------------------

        self.m = m
        self.debug = debug_mode

        if not self.debug:
            warnings.filterwarnings("ignore")

        self.select = select
        self.n_p = n_p

        self.timeout = timeout
        # self.use_numba = use_numba # Numba likely incompatible with dict structure without rework
        self.use_numba = False  # Force disable Numba for now
        print("Warning: Numba decorator usage disabled for Scheme C dictionary structure.")

    # code2file is likely not needed/compatible anymore
    # def code2file(self,code):
    #     with open("./ael_alg.py", "w") as file:
    #     # Write the code to the file
    #         file.write(code)
    #     return

    def add2pop(self, population, offspring):
        """Adds offspring if objective is not duplicated."""
        # Consider adding a check if offspring is None
        if offspring is None or offspring.get('objective') is None:
            print(
                "Warning: Attempted to add invalid offspring (None or no objective). Skipping.")
             return False
        for ind in population:
            # Handle potential None objectives during comparison
            if ind.get('objective') is not None and ind['objective'] == offspring['objective']:
                if self.debug:
                    print(
                        f"Duplicated objective ({offspring['objective']}), not adding.")
                return False
        population.append(offspring)
        return True

    def check_duplicate(self, population, offspring_dict):
        """Checks if an identical offspring dictionary already exists (simple check)."""
        # This is a basic check comparing all code parts. Might be slow or need refinement.
        # Currently NOT used in get_offspring.
        if offspring_dict is None:
             return False
        keys_to_check = ["hica", "ghop", "eoss"]
        if not all(key in offspring_dict for key in keys_to_check):
            return False  # Invalid dict format

        for ind in population:
             if all(key in ind for key in keys_to_check) and \
                ind["hica"] == offspring_dict["hica"] and \
                ind["ghop"] == offspring_dict["ghop"] and \
                ind["eoss"] == offspring_dict["eoss"]:
                 return True
        return False

    def population_generation(self):
        """Generates the initial population using the i1 operator (Generation 0)."""
        n_create = self.pop_size  # Aim to create full pop size initially
        population = []
        attempts = 0
        max_attempts = n_create * 2  # Allow some failures

        print(
            f"Attempting to generate initial population (target size: {n_create}, Generation 0)...")
        while len(population) < n_create and attempts < max_attempts:
            attempts += 1
            # Pass generation=0 to get_algorithm for initial population
            _, offsprings = self.get_algorithm(
                [], 'i1', generation=0)  # Pass generation=0
            added_count = 0
            for p_dict in offsprings:
                # Add only valid offspring with objective
                if p_dict and p_dict.get('objective') is not None:
                    # Use add2pop which checks for objective duplication
                    if self.add2pop(population, p_dict):
                        added_count += 1
                        if len(population) >= n_create:
                            break  # Stop if target size reached
            if self.debug:
                print(
                    f"  Generation 0 attempt {attempts}: Added {added_count} new individuals.")
            if len(population) >= n_create:
                break

        if len(population) < n_create:
            print(
                f"Warning: Could only generate {len(population)}/{n_create} initial individuals (Gen 0) after {attempts} attempts.")
        else:
            print(
                f"Successfully generated initial population (Gen 0) of size {len(population)}.")
        return population

    def population_generation_seed(self, seeds: list[dict], n_p: int):
        """Generates population from seeds (Generation 0), assuming seeds are dictionaries."""
        population = []
        if not seeds:
            return population

        print(
            f"Evaluating {len(seeds)} seed individuals (Generation 0) using {n_p} processes...")
        # Assuming seeds are already in the format {"hica":..., "ghop":..., "eoss":...}
        try:
             # Evaluate seeds in parallel, passing the whole dictionary, generation=0, AND worker_id
             results = Parallel(n_jobs=n_p)(
                delayed(self.interface_eval.evaluate)(
                    seed, generation=0, worker_id=i, num_gpus=n_p)
                 for i, seed in enumerate(seeds)
             )
        except Exception as e:
             print(f"Error during parallel evaluation of seeds (Gen 0): {e}")
            return population  # Return empty list on parallel failure

        print("Seed evaluation (Gen 0) finished. Building population...")
        for i, seed_dict in enumerate(seeds):
            try:
                fitness = results[i]
                # Create a new dictionary to store in population (avoid modifying original seeds)
                # Ensure seed_dict is actually a dict before copying
                if isinstance(seed_dict, dict):
                    # Use deepcopy if seeds might contain mutable objects
                    evaluated_seed = copy.deepcopy(seed_dict)
                    evaluated_seed['objective'] = np.round(fitness, 5)
                    # Add other necessary keys if missing, though evaluate should handle dict format
                    # Indicate seeded in Gen 0
                    evaluated_seed.setdefault('algorithm', 'seeded_g0')
                    # Store generation info
                    evaluated_seed.setdefault('other_inf', {'generation': 0})
                    population.append(evaluated_seed)
                else:
                    print(
                        f"Warning: Seed element at index {i} is not a dictionary. Skipping.")

            except Exception as e:
                print(
                    f"Error processing seed result at index {i} (Gen 0): {e}")
                # Optionally continue or stop based on severity
                # continue

        print(
            f"Initialization from seeds (Gen 0) finished! Created {len(population)} individuals.")
        return population

    def _get_alg(self, pop: list[dict], operator: str) -> tuple[list[dict] | None, dict | None]:
        """Internal method to call the correct Evolution method."""
        offspring_dict = None  # Will hold {"hica":..., "ghop":..., "eoss":...}
        parents = None
        try:
            if operator == "i1":
                parents = None  # No parents for initial generation
                offspring_dict = self.evol.i1()  # Returns dict or None
            elif operator in ["e1", "e2"]:
                if not pop:
                    raise ValueError(
                        f"Operator {operator} requires existing population.")
                parents = self.select.parent_selection(
                    pop, self.m)  # Select returns list of dicts
                if not parents:
                    raise ValueError("Parent selection returned empty list.")
                # Pass the list of parent dicts to the evolution method
                offspring_dict = getattr(self.evol, operator)(
                    parents)  # Call self.evol.e1(parents) or e2
            elif operator in ["m1", "m2", "m3"]:
                if not pop:
                    raise ValueError(
                        f"Operator {operator} requires existing population.")
                parents = self.select.parent_selection(
                    pop, 1)  # Select returns list of size 1
                if not parents:
                    raise ValueError("Parent selection returned empty list.")
                # Pass the single parent dictionary
                # Call self.evol.m1(parent_dict) etc.
                offspring_dict = getattr(self.evol, operator)(parents[0])
            else:
                print(
                    f"Evolution operator [{operator}] has not been implemented or is unknown!")
                return None, None  # Return None, None for unknown operator

            # offspring_dict will be None if self.evol.* returned None (LLM/parsing error)
            if offspring_dict is None:
                print(
                    f"Warning: Operator {operator} failed to generate a valid offspring dictionary.")
                 return parents, None

        except Exception as e:
            print(f"Error during _get_alg for operator {operator}: {e}")
            traceback.print_exc()  # Print stack trace for debugging
            return parents, None  # Return None offspring on error

        # Return parent(s) used and the new offspring dict
        return parents, offspring_dict

    def get_offspring(self, pop: list[dict], operator: str, generation: int) -> tuple[list[dict] | None, dict | None]:
        """Generates AND evaluates a single offspring using the specified operator for a given generation."""
        parents = None
        offspring_dict = None
        try:
            # 1. Get unevaluated offspring dictionary
            parents, offspring_dict = self._get_alg(pop, operator)

            if offspring_dict is None:
                print(
                    f"DEBUG: Skipping evaluation because offspring_dict is None for operator {operator} (Gen {generation}).")
                return parents, None

            # Explicitly check if it's a dictionary before proceeding
            if not isinstance(offspring_dict, dict):
                print(
                    f"ERROR: offspring_dict is NOT a dict right before evaluation! (Gen {generation}). Type: {type(offspring_dict)}. Value: {offspring_dict}")
                 return parents, None

            # --- !! PRINT GENERATED/FIXED CODE !! ---
            print(f"\n" + "="*20 +
                  f" Code for Offspring (Operator: {operator}, Gen: {generation}) " + "="*20)
            is_hica_fixed = 'hica' in self.evol.fixed_modules_code  # Check if HICA is fixed
            is_ghop_fixed = 'ghop' in self.evol.fixed_modules_code  # Check if GHOP is fixed
            is_eoss_fixed = 'eoss' in self.evol.fixed_modules_code  # Check if EOSS is fixed

            print(
                f"--- HICA Code {'(Fixed)' if is_hica_fixed else '(Generated by LLM)'} ---")
            print(offspring_dict.get('hica', '!!! HICA code missing !!!'))
            print("-----------------")

            print(
                f"--- GHOP Code (Method Body) {'(Fixed)' if is_ghop_fixed else '(Generated by LLM)'} ---")
            print(offspring_dict.get('ghop', '!!! GHOP code missing !!!'))
            print("-----------------------------")

            print(
                f"--- EOSS Code (Method Body) {'(Fixed)' if is_eoss_fixed else '(Generated by LLM)'} ---")
            # Avoid printing potentially very long fixed EOSS code every time
            if is_eoss_fixed:
                print("[Fixed EOSS code - Content omitted for brevity]")
            else:
                 print(offspring_dict.get('eoss', '!!! EOSS code missing !!!'))
            print("-----------------------------")
            print("="*70 + "\n")
            # --- !! END CODE PRINT !! ---

            # 3. Evaluate (If offspring_dict is not None and is a dict)
            fitness = None
            try:
                with concurrent.futures.ThreadPoolExecutor() as executor:
                    # Pass the generation number to evaluate
                    future = executor.submit(
                        self.interface_eval.evaluate, offspring_dict, generation=generation)  # Pass generation
                    fitness = future.result(timeout=self.interface_eval.EVALUATION_TIMEOUT if hasattr(
                        self.interface_eval, 'EVALUATION_TIMEOUT') else self.timeout)  # Use problem timeout if available

            except concurrent.futures.TimeoutError:
                eval_timeout = self.interface_eval.EVALUATION_TIMEOUT if hasattr(
                    self.interface_eval, 'EVALUATION_TIMEOUT') else self.timeout
                print(
                    f"Evaluation timed out after {eval_timeout} seconds for operator {operator} (Gen {generation}).")
                fitness = getattr(self.interface_eval,
                                  'FITNESS_PENALTY', 9999.0) + 100

            except Exception as eval_e:
                print(
                    f"Error during evaluation execution (Gen {generation}): {eval_e}")
                fitness = getattr(self.interface_eval,
                                  'FITNESS_PENALTY', 9999.0) + 200
                 # traceback.print_exc() # Uncomment if needed

            # 5. Assign fitness to the dictionary
            if fitness is not None:
                if isinstance(offspring_dict, dict):
                    offspring_dict['objective'] = np.round(fitness, 5)
                    # Optionally store generation info in the offspring dict itself
                    if 'other_inf' not in offspring_dict or offspring_dict['other_inf'] is None:
                        offspring_dict['other_inf'] = {}
                    # Check if other_inf is a dict
                    if isinstance(offspring_dict['other_inf'], dict):
                        offspring_dict['other_inf']['generation'] = generation
                else:
                    print(
                        f"ERROR: offspring_dict became non-dict before fitness assignment! (Gen {generation})")
                     return parents, None
            else:
                print(
                    f"Warning: Evaluation finished with None fitness (Gen {generation}). Assigning high penalty.")
                if isinstance(offspring_dict, dict):
                    offspring_dict['objective'] = getattr(
                        self.interface_eval, 'FITNESS_PENALTY', 9999.0) + 300
                     # Optionally store generation info even on failure
                    if 'other_inf' not in offspring_dict or offspring_dict['other_inf'] is None:
                        offspring_dict['other_inf'] = {}
                    if isinstance(offspring_dict['other_inf'], dict):
                        offspring_dict['other_inf']['generation'] = generation
                else:
                    print(
                        f"ERROR: offspring_dict was non-dict and fitness was None! (Gen {generation})")
                     return parents, None

        except Exception as e:
            print(
                f"Unexpected error in get_offspring for operator {operator} (Gen {generation}): {e}")
            traceback.print_exc()
            # Try to assign penalty and generation even on outer error if dict exists
            if offspring_dict and isinstance(offspring_dict, dict):
                 if 'objective' not in offspring_dict:
                    offspring_dict['objective'] = getattr(
                        self.interface_eval, 'FITNESS_PENALTY', 9999.0) + 400
                if 'other_inf' not in offspring_dict or offspring_dict['other_inf'] is None:
                    offspring_dict['other_inf'] = {}
                if isinstance(offspring_dict['other_inf'], dict):
                    offspring_dict['other_inf']['generation'] = generation
            elif offspring_dict is None:
                return parents, None  # Cannot assign if dict is None

        return parents, offspring_dict

    def get_algorithm(self, pop: list[dict], operator: str, generation: int) -> tuple[list[list[dict] | None], list[dict | None]]:
        """Generates AND evaluates a list of offspring using the specified operator for a given generation."""
        n_create = self.pop_size
        all_parents = []
        offspring_list = []  # List to store evaluated offspring dicts
        # List to store unevaluated dicts before parallel eval
        unevaluated_offspring_dicts = []

        # 1. Generate unevaluated offspring dictionaries
        print(
            f"Generating {n_create} offspring candidate(s) for Gen {generation} using operator {operator}...")
        generation_attempts = 0
        max_generation_attempts = n_create * 3  # Allow more attempts
        while len(unevaluated_offspring_dicts) < n_create and generation_attempts < max_generation_attempts:
            generation_attempts += 1
            parents, offspring_dict = self._get_alg(pop, operator)
            if offspring_dict is not None and isinstance(offspring_dict, dict):
                # Store the parent(s) and the unevaluated offspring dict
                # We need a way to link parents back after evaluation, maybe add temp ID?
                offspring_dict['_temp_eval_id'] = len(
                    unevaluated_offspring_dicts)  # Add temporary ID
                unevaluated_offspring_dicts.append(offspring_dict)
                all_parents.append(parents)  # Store corresponding parents
            else:
                print(
                    f"Warning: Failed to generate valid offspring dict on attempt {generation_attempts} for operator {operator}, Gen {generation}.")
            if len(unevaluated_offspring_dicts) >= n_create:
                break

        if not unevaluated_offspring_dicts:
            print(
                f"ERROR: Failed to generate ANY valid offspring dictionaries after {generation_attempts} attempts for Gen {generation}. Returning empty list.")
             return [], []
        if len(unevaluated_offspring_dicts) < n_create:
            print(
                f"Warning: Generated only {len(unevaluated_offspring_dicts)}/{n_create} offspring for Gen {generation}.")

        # 2. Evaluate generated offspring in parallel
        print(
            f"Starting parallel evaluation for {len(unevaluated_offspring_dicts)} generated offspring (Gen {generation}) using {self.n_p} processes...")
        evaluated_fitness = []
        try:
            evaluated_fitness = Parallel(n_jobs=self.n_p)(
                delayed(self.interface_eval.evaluate)(
                    offspring,           # Pass the unevaluated dict
                    generation=generation,  # Pass generation
                    worker_id=i,         # Pass the worker ID (index)
                    num_gpus=self.n_p     # Pass total number of processes/gpus
                )
                for i, offspring in enumerate(unevaluated_offspring_dicts)
            )
        except Exception as eval_e:
            print(
                f"ERROR: Parallel evaluation failed for Gen {generation}: {eval_e}")
            traceback.print_exc()
            # Return empty evaluated list, but keep parents? Or return None, None?
            # Let's return the parents found so far, and an empty offspring list.
            return all_parents, []

        print(f"Parallel evaluation finished for Gen {generation}.")

        # 3. Combine evaluated fitness with offspring dictionaries
        offspring_list = []
        for i, offspring_dict in enumerate(unevaluated_offspring_dicts):
            if i < len(evaluated_fitness):
                fitness = evaluated_fitness[i]
                # Check for None fitness (might happen if evaluate had internal error but didn't crash Parallel)
                if fitness is None:
                    print(
                        f"Warning: Evaluation returned None fitness for offspring index {i}, Gen {generation}. Assigning penalty.")
                    fitness = getattr(self.interface_eval,
                                      'FITNESS_PENALTY', 9999.0) + 500

                # Add fitness to the dictionary (use the _temp_eval_id if needed, though index i should match)
                offspring_dict['objective'] = np.round(fitness, 5)
                offspring_dict.setdefault('algorithm', operator)
                offspring_dict.setdefault(
                    'other_inf', {'generation': generation})
                if '_temp_eval_id' in offspring_dict:
                    del offspring_dict['_temp_eval_id']  # Clean up temp ID
                offspring_list.append(offspring_dict)
            else:
                # This case should ideally not happen if Parallel returns results for all inputs
                print(
                    f"Warning: Missing fitness result for offspring index {i}, Gen {generation}.")
                # Optionally create a dummy offspring with penalty fitness?
                # offspring_dict['objective'] = getattr(self.interface_eval, 'FITNESS_PENALTY', 9999.0) + 600
                # offspring_list.append(offspring_dict)

        # Return the list of parents used and the list of evaluated offspring
        return all_parents, offspring_list
