import numpy as np
import json
import random
import time
import os  # Import os for path joining if needed later
import copy  # Import copy module

from .eoh_interface_EC import InterfaceEC
# main class for eoh


class EOH:

    # initilization
    def __init__(self, paras, problem, select, manage, **kwargs):

        self.prob = problem
        self.select = select
        self.manage = manage
        
        # LLM settings
        self.use_local_llm = paras.llm_use_local
        self.llm_local_url = paras.llm_local_url
        self.api_endpoint = paras.llm_api_endpoint  # currently only API2D + GPT
        self.api_key = paras.llm_api_key
        self.llm_model = paras.llm_model

        # ------------------ RZ: use local LLM ------------------
        # self.use_local_llm = kwargs.get('use_local_llm', False)
        # assert isinstance(self.use_local_llm, bool)
        # if self.use_local_llm:
        #     assert 'url' in kwargs, 'The keyword "url" should be provided when use_local_llm is True.'
        #     assert isinstance(kwargs.get('url'), str)
        #     self.url = kwargs.get('url')
        # -------------------------------------------------------

        # Experimental settings       
        # popopulation size, i.e., the number of algorithms in population
        self.pop_size = paras.ec_pop_size
        self.n_pop = paras.ec_n_pop  # number of populations

        self.operators = paras.ec_operators
        self.operator_weights = paras.ec_operator_weights
        if paras.ec_m > self.pop_size or paras.ec_m <= 1:  # Corrected condition m=1 is invalid
            print(
                f"Warning: m ({paras.ec_m}) should be > 1 and <= pop_size ({self.pop_size}). Adjusting m to 2.")
            paras.ec_m = 2  # Ensure m is at least 2 for operators needing multiple parents
        self.m = paras.ec_m

        self.debug_mode = paras.exp_debug_mode  # if debug
        self.ndelay = 1  # default

        self.use_seed = paras.exp_use_seed
        self.seed_path = paras.exp_seed_path
        self.load_pop = paras.exp_use_continue
        self.load_pop_path = paras.exp_continue_path
        self.load_pop_id = paras.exp_continue_id

        self.output_path = paras.exp_output_path
        # Ensure results directories exist
        os.makedirs(os.path.join(self.output_path,
                    "results", "pops"), exist_ok=True)
        os.makedirs(os.path.join(self.output_path,
                    "results", "pops_best"), exist_ok=True)
        # os.makedirs(os.path.join(self.output_path, "results", "history"), exist_ok=True) # If history logging is re-enabled

        self.exp_n_proc = paras.exp_n_proc
        
        self.timeout = paras.eva_timeout

        # self.use_numba = paras.eva_numba_decorator # Numba likely incompatible
        self.use_numba = False  # Force disable for Scheme C

        # Track the best individual found across all generations
        self.global_best_individual = None
        # Track the best fitness found so far
        self.global_best_fitness = -float('inf')

        print("- EoH parameters loaded -")

        # Set a random seed
        # Consider making the seed configurable via paras
        random.seed(paras.exp_seed if hasattr(paras, 'exp_seed') else 2024)
        np.random.seed(paras.exp_seed if hasattr(paras, 'exp_seed') else 2024)

    # add new individual to population
    def add2pop(self, population: list[dict], offsprings: list[dict | None]):
        """
        Adds valid offspring (non-None with objective) to the population,
        checking for objective duplication.
        """
        added_count = 0
        for off in offsprings:
            # Skip invalid offspring (None or missing objective)
            if off is None or off.get('objective') is None:
                continue

            is_duplicate = False
            for ind in population:
                # Check if existing individual also has an objective before comparing
                if ind.get('objective') is not None and ind['objective'] == off['objective']:
                    if self.debug_mode:
                        print(
                            f"Duplicated objective ({off['objective']}), skipping.")
                    is_duplicate = True
                    break  # Found duplicate objective, no need to check further

            if not is_duplicate:
                population.append(off)
                added_count += 1
        if self.debug_mode: print(
            f"Added {added_count} new individuals to population.")

    # run eoh 
    def run(self):

        print("- Evolution Start -")
        print(f"  Population Size: {self.pop_size}, Generations: {self.n_pop}")
        print(
            f"  Operators: {self.operators}, Weights: {self.operator_weights}")
        print(f"  Parallel Procs: {self.exp_n_proc}, Timeout: {self.timeout}s")

        time_start = time.time()

        # interface for large language model (llm)
        # interface_llm = PromptLLMs(self.api_endpoint,self.api_key,self.llm_model,self.debug_mode)

        # interface for evaluation
        interface_prob = self.prob

        # interface for ec operators
        interface_ec = InterfaceEC(self.pop_size, self.m, self.api_endpoint, self.api_key, self.llm_model, self.use_local_llm, self.llm_local_url,
                                   self.debug_mode, interface_prob, select=self.select, n_p=self.exp_n_proc,
                                   timeout=self.timeout, use_numba=self.use_numba
                                   )

        # initialization
        population = []
        n_start = 0  # Generation counter start
        if self.load_pop:  # load population from files
             load_file = os.path.join(
                 self.load_pop_path, f"population_generation_{self.load_pop_id}.json")
             print(f"Attempting to load initial population from {load_file}")
             try:
                 with open(load_file) as file:
                     population = json.load(file)
                 print(
                     f"Initial population ({len(population)} individuals) loaded successfully!")
                 n_start = self.load_pop_id  # Start from the next generation id
                 # --- ADDED: Initialize global best from loaded population ---
                 if population:
                     for ind in population:
                         if ind is not None and 'objective' in ind:
                             current_fitness = ind.get('objective')
                             if isinstance(current_fitness, (int, float)):
                                 if current_fitness > self.global_best_fitness:
                                     self.global_best_fitness = current_fitness
                                     self.global_best_individual = copy.deepcopy(
                                         ind)
                     if self.global_best_individual:
                        print(
                            f"Initialized Global Best from loaded pop: Obj {self.global_best_fitness:.5f}")
                 # -----------------------------------------------------------
             except FileNotFoundError:
                 print(f"Error: Load population file not found: {load_file}")
                 print("Starting with empty population.")
                 self.load_pop = False  # Fall back to creating new pop
             except json.JSONDecodeError:
                 print(f"Error: Could not decode JSON from file: {load_file}")
                 print("Starting with empty population.")
                 self.load_pop = False  # Fall back to creating new pop
             except Exception as e:
                  print(f"Error loading population: {e}")
                  self.load_pop = False

        # If not loading or loading failed, create new pop
        if not self.load_pop:
             if self.use_seed:
                 print(
                     f"Attempting to generate population from seed file: {self.seed_path}")
                 try:
                     with open(self.seed_path) as file:
                         # Assumes seed file contains list of dicts
                         seed_data = json.load(file)
                     # population_generation_seed now handles evaluation and returns list of dicts
                     population = interface_ec.population_generation_seed(
                         seed_data, self.exp_n_proc)
                     # --- ADDED: Initialize global best from seeded population ---
                     if population:
                         for ind in population:
                             if ind is not None and 'objective' in ind:
                                 current_fitness = ind.get('objective')
                                 if isinstance(current_fitness, (int, float)):
                                     if current_fitness > self.global_best_fitness:
                                         self.global_best_fitness = current_fitness
                                         self.global_best_individual = copy.deepcopy(
                                             ind)
                         if self.global_best_individual:
                             print(
                                 f"Initialized Global Best from seeds: Obj {self.global_best_fitness:.5f}")
                     # -----------------------------------------------------------
                 except FileNotFoundError:
                      print(f"Error: Seed file not found: {self.seed_path}")
                      print("Starting with random generation.")
                      self.use_seed = False  # Fall back to random
                 except json.JSONDecodeError:
                      print(
                          f"Error: Could not decode JSON from seed file: {self.seed_path}")
                      self.use_seed = False
                 except Exception as e:
                     print(f"Error processing seed data: {e}")
                     self.use_seed = False

             # If not using seed or seeding failed, use random generation
             if not self.use_seed:
                 print("Creating initial population randomly:")
                 # population_generation calls get_algorithm('i1') which evaluates
                 population = interface_ec.population_generation()
                 # Apply management *after* initial generation attempt
                 population = self.manage.population_management(
                     population, self.pop_size)
                 # --- ADDED: Initialize global best from random generation ---
                 if population:
                     for ind in population:
                         if ind is not None and 'objective' in ind:
                             current_fitness = ind.get('objective')
                             if isinstance(current_fitness, (int, float)):
                                 if current_fitness > self.global_best_fitness:
                                     self.global_best_fitness = current_fitness
                                     self.global_best_individual = copy.deepcopy(
                                         ind)
                     if self.global_best_individual:
                         print(
                             f"Initialized Global Best from initial random pop: Obj {self.global_best_fitness:.5f}")
                 # -------------------------------------------------------------

             # Save the newly created initial population (gen 0)
             print(f"Initial population size: {len(population)}")
             if population:  # Only save if population is not empty
                 print("Initial Pop Objs: ", end=" ")
                 for ind in population: print(
                     f"{ind.get('objective', 'N/A')} ", end="")
                 print()
                 filename = os.path.join(
                     self.output_path, "results", "pops", "population_generation_0.json")
                 try:
                     with open(filename, 'w') as f:
                         # Use indent=4 for readability
                         json.dump(population, f, indent=4)
                     print(f"Saved initial population to {filename}")
                 except Exception as e:
                      print(f"Error saving initial population: {e}")
             else:
                 print("Warning: Initial population is empty after generation/seeding.")
             n_start = 0  # Ensure generation counter starts at 0

        # main loop
        n_op = len(self.operators)

        # pop represents the current generation index (0-based)
        for pop in range(n_start, self.n_pop):
            # Use 1-based generation for user display/logs if preferred, but pass 0-based pop internally
            current_generation = pop + 1
            print(
                f"\n--- Starting Generation {current_generation} / {self.n_pop} ---")
            # print(f" [{na + 1} / {self.pop_size}] ", end="|")
            for i in range(n_op):  # Loop through operators
                op = self.operators[i]
                print(f" OP: {op}, [{i + 1} / {n_op}] ", end="|") 
                op_w = self.operator_weights[i]
                if (np.random.rand() < op_w):
                    # Call interface_ec.get_algorithm, passing the current generation index 'pop'
                    parents, offsprings = interface_ec.get_algorithm(
                        population, op, generation=pop)  # Pass generation index
                    # Add valid offspring to population
                    self.add2pop(population, offsprings)
                    # Print objective of newly added offspring
                    valid_offs = [off for off in offsprings if off and off.get(
                        'objective') is not None]
                    if valid_offs:
                        print(f" Objs({len(valid_offs)} added): ", end="")
                        for off in valid_offs: print(
                            f"{off.get('objective', 'N/A')} ", end="")
                        print("|", end="")
                    else:
                        print(" No valid offspring added |", end="")

                else:  # Operator skipped based on weight
                     print(" Skipped |", end="")

                # populatin management after applying the operator
                size_act = min(len(population), self.pop_size)
                population = self.manage.population_management(
                    population, size_act)
                print()  # Newline after operator info

            # Save population to a file after all operators for the generation are done
            filename = os.path.join(self.output_path, "results", "pops",
                                    f"population_generation_{current_generation}.json")
            try:
                # --- ADDED: Sort population by objective (descending) before saving --- 
                # Handle potential None or non-numeric objectives gracefully during sorting
                sorted_population = sorted(
                    population,
                    key=lambda ind: ind.get('objective') if isinstance(
                        ind.get('objective'), (int, float)) else -float('inf'),
                    reverse=True  # Higher objective is better
                )
                # ---------------------------------------------------------------------
                with open(filename, 'w') as f:
                    # Save the sorted population
                    json.dump(sorted_population, f, indent=4)
            except Exception as e:
                print(
                    f"Warning: Failed to save population for generation {current_generation}: {e}")

            # Save the best one by finding the max objective
            if population:  # Check if population is not empty
                try:
                    # Find the individual with the maximum objective value
                    # Handle potential None or non-numeric objectives gracefully during comparison
                    best_individual = max(population, key=lambda ind: ind.get(
                        'objective') if isinstance(ind.get('objective'), (int, float)) else -float('inf'))

                    # Ensure the best individual actually has a valid objective before saving
                    if 'objective' in best_individual and isinstance(best_individual['objective'], (int, float)):
                        filename = os.path.join(
                            self.output_path, "results", "pops_best", f"population_generation_{current_generation}.json")
                try:
                    with open(filename, 'w') as f:
                                json.dump(best_individual, f, indent=4)
                            # Added print statement for clarity
                            print(f"Saved best individual (Obj: {best_individual['objective']:.5f}) for generation {current_generation}.")
                except Exception as e:
                    print(f"Warning: Failed to save best individual for generation {current_generation}: {e}")
                    else:
                        print(f"Warning: Could not find a best individual with a valid objective in generation {current_generation}.")

                except ValueError: # Handles case where population might be empty after filtering?
                     print(f"Warning: Population is empty or contains no individuals with valid objectives, cannot save best individual for generation {current_generation}.")

            else:
                 print(f"Warning: Population is empty after management, cannot save best individual for generation {current_generation}.")

            print(f"--- Generation {current_generation} of {self.n_pop} finished. Time Cost:  {((time.time()-time_start)/60):.1f} m")
            if population:
                print("Pop Objs: ", end=" ")
                for i in range(len(population)):
                    # Ensure objective exists before trying to access it
                    obj_val = population[i].get('objective', 'N/A')
                    print(str(obj_val) + " ", end="")
                print()
            else:
                 print("Population is empty.")

        print("\n- Evolution Finished -")

        # --- ADDED: Save the overall global best individual found during the run ---
        if self.global_best_individual is not None:
            global_best_filename = os.path.join(self.output_path, "results", "global_pop_best.json")
            try:
                # Ensure the 'results' directory exists (should already exist, but good practice)
                os.makedirs(os.path.join(self.output_path, "results"), exist_ok=True)
                with open(global_best_filename, 'w') as f:
                    json.dump(self.global_best_individual, f, indent=4)
                print(f"Saved Global Best Individual (Obj: {self.global_best_fitness:.5f}) to {global_best_filename}")
            except Exception as e:
                print(f"Warning: Failed to save global best individual: {e}")
        else:
            print("Warning: No global best individual was found or tracked during the run.")
        # -------------------------------------------------------------------------

