import re
import time
from ...llm.interface_LLM import InterfaceLLM
import traceback # Import traceback for better error logging

# Delimiter constants are no longer needed here
# from .....examples.MOOSF.MOOSFCombinedProblem.moosf_combined_prompts import ...

class Evolution():

    def __init__(self, api_endpoint, api_key, model_LLM, llm_use_local, llm_local_url, debug_mode, prompts, **kwargs):
        """
        Initializes the Evolution class. Fixed code is now retrieved from prompts object.
        """
        self.prompts = prompts
        # Retrieve fixed code from prompts object
        self.fixed_modules_code = self.prompts.fixed_modules_code if hasattr(self.prompts, 'fixed_modules_code') else {}

        # --- Print which modules are fixed ---
        print("- Evolution Class Initialized -")
        if self.fixed_modules_code:
            print(f"  Modules fixed to provided code: {list(self.fixed_modules_code.keys())}")
        else:
            print("  No modules are fixed, all will be generated by LLM.")
        # --------------------------------------

        # Dummy attributes
        self.prompt_func_name    = self.prompts.get_func_name()
        self.prompt_func_inputs  = self.prompts.get_func_inputs()
        self.prompt_func_outputs = self.prompts.get_func_outputs()

        # LLM settings
        self.api_endpoint = api_endpoint
        self.api_key = api_key
        self.model_LLM = model_LLM
        self.debug_mode = debug_mode

        self.interface_llm = InterfaceLLM(self.api_endpoint, self.api_key, self.model_LLM, llm_use_local, llm_local_url, self.debug_mode)
        # print("- Evolution Class Initialized for Scheme C (Separate Calls) -") # Modified print above

    # Removed old delimiter parsing methods: _parse_delimited_code, _get_and_parse_combined_code

    def _get_llm_response_with_retry(self, prompt_content: str, max_retries: int = 2) -> str | None:
        """Gets response from LLM with basic retry logic."""
        response = None
        n_retry = 0
        while n_retry <= max_retries:
            if n_retry > 0:
                print(f"Retrying LLM call (Attempt {n_retry}/{max_retries})...")
                time.sleep(1)
            response = self.interface_llm.get_response(prompt_content)
            if response:
                return response.strip()
            print(f"LLM call failed or returned empty on attempt {n_retry}.")
            n_retry += 1
        print(f"Failed to get valid LLM response after {max_retries + 1} attempts.")
        return None

    def _extract_code_from_response(self, response: str | None) -> str | None:
        """
        Basic extraction/cleanup for LLM response assuming it mainly contains code.
        Removes potential ```python ... ``` blocks if present.
        Returns None if input is None.
        """
        if response is None:
            return None

        # Remove potential markdown code blocks
        response = re.sub(r'^```python\s*', '', response, flags=re.MULTILINE)
        response = re.sub(r'\s*```$', '', response, flags=re.MULTILINE)
        cleaned_code = response.strip()

        # Add more specific cleaning if needed based on observed LLM behavior
        # For example, removing leading/trailing explanations if the LLM doesn't follow instructions.

        return cleaned_code if cleaned_code else None # Return None if cleaning results in empty string

    def _generate_individual_modules(self, parent_dict: dict | None = None, fixed_modules: dict | None = None) -> dict | None:
        """
        Generates HICA, GHOP, EOSS code. Uses fixed code if provided, otherwise calls LLM.
        Uses parent code as fallback if LLM generation fails. Logs source for each module.
        Returns a dictionary {'hica': ..., 'ghop': ..., 'eoss': ...} or None if critical generation fails.
        """
        is_initial_generation = parent_dict is None
        child_dict = {}
        fixed_modules = fixed_modules if fixed_modules else {} # Ensure it's a dict
        # Flags to track generation source
        hica_source = "Fixed" if "hica" in fixed_modules else "LLM_Attempt"
        ghop_source = "Fixed" if "ghop" in fixed_modules else "LLM_Attempt"
        eoss_source = "Fixed" if "eoss" in fixed_modules else "LLM_Attempt"

        # --- Generate HICA ---
        if hica_source == "Fixed":
            child_dict["hica"] = fixed_modules["hica"]
            print("DEBUG: Using fixed code for HICA.")
        else:
            parent_hica = parent_dict.get("hica") if parent_dict else None
            hica_prompt = self.prompts.get_hica_mutation_prompt(parent_hica_code=parent_hica)
            print(f"DEBUG: Generating HICA {'from scratch' if is_initial_generation else 'mutation'} via LLM...")
            hica_response = self._get_llm_response_with_retry(hica_prompt)
            new_hica_code = self._extract_code_from_response(hica_response)

            if new_hica_code:
                child_dict["hica"] = new_hica_code
                hica_source = "LLM_Success" # Mark as successful LLM generation
                if self.debug_mode: print("DEBUG: HICA generated successfully by LLM.")
            elif not is_initial_generation and parent_hica:
                child_dict["hica"] = parent_hica # Fallback to parent
                hica_source = "ParentFallback" # Mark as fallback
                print(">>> WARNING: HICA generation failed, FALLBACK TO PARENT code.") # Log fallback
            else:
                print("Error: Initial HICA generation failed (and not fixed).")
                return None

        # --- Generate GHOP ---
        if ghop_source == "Fixed":
            child_dict["ghop"] = fixed_modules["ghop"]
            print("DEBUG: Using fixed code for GHOP.")
        else:
            parent_ghop = parent_dict.get("ghop") if parent_dict else None
            ghop_prompt = self.prompts.get_ghop_mutation_prompt(parent_ghop_code=parent_ghop)
            print(f"DEBUG: Generating GHOP {'from scratch' if is_initial_generation else 'mutation'} via LLM...")
            ghop_response = self._get_llm_response_with_retry(ghop_prompt)
            new_ghop_code = self._extract_code_from_response(ghop_response)

            if new_ghop_code:
                child_dict["ghop"] = new_ghop_code
                ghop_source = "LLM_Success" # Mark as successful LLM generation
                if self.debug_mode: print("DEBUG: GHOP generated successfully by LLM.")
            elif not is_initial_generation and parent_ghop:
                child_dict["ghop"] = parent_ghop # Fallback to parent
                ghop_source = "ParentFallback" # Mark as fallback
                print(">>> WARNING: GHOP generation failed, FALLBACK TO PARENT code.") # Log fallback
            else:
                print("Error: Initial GHOP generation failed (and not fixed).")
                return None

        # --- Generate EOSS ---
        if eoss_source == "Fixed":
            child_dict["eoss"] = fixed_modules["eoss"]
            print("DEBUG: Using fixed code for EOSS.")
        else:
            parent_eoss = parent_dict.get("eoss") if parent_dict else None
            eoss_prompt = self.prompts.get_eoss_mutation_prompt(parent_eoss_code=parent_eoss)
            print(f"DEBUG: Generating EOSS {'from scratch' if is_initial_generation else 'mutation'} via LLM...")
            eoss_response = self._get_llm_response_with_retry(eoss_prompt)
            new_eoss_code = self._extract_code_from_response(eoss_response)

            if new_eoss_code:
                child_dict["eoss"] = new_eoss_code
                eoss_source = "LLM_Success" # Mark as successful LLM generation
                if self.debug_mode: print("DEBUG: EOSS generated successfully by LLM.")
            elif not is_initial_generation and parent_eoss:
                child_dict["eoss"] = parent_eoss # Fallback to parent
                eoss_source = "ParentFallback" # Mark as fallback
                print(">>> WARNING: EOSS generation failed, FALLBACK TO PARENT code.") # Log fallback
            else:
                print("Error: Initial EOSS generation failed (and not fixed).")
                return None

        # Log the final source of each component
        print(f"DEBUG: Finished generating modules. Sources: [HICA:{hica_source}, GHOP:{ghop_source}, EOSS:{eoss_source}]")
        return child_dict

    # --- Public Methods for Variation Operators ---
    # Pass self.fixed_modules_code to _generate_individual_modules

    def i1(self) -> dict | None:
        """Initial generation using separate calls or fixed code."""
        if self.debug_mode: print("\n >>> Generating initial individual using [ i1 ]...")
        child_dict = self._generate_individual_modules(parent_dict=None, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed initial code dictionary: \n", child_dict)
        return child_dict

    def e1(self, parents: list[dict]) -> dict | None:
        """Exploration variation using separate calls or fixed code."""
        if not parents: print("Error [e1]: No parents provided."); return None
        parent_dict = parents[0]
        if self.debug_mode: print("\n >>> Generating individual using [ e1 ]...")
        child_dict = self._generate_individual_modules(parent_dict=parent_dict, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed code dictionary: \n", child_dict)
        return child_dict

    def e2(self, parents: list[dict]) -> dict | None:
        """Exploitation variation using separate calls or fixed code."""
        if not parents: print("Error [e2]: No parents provided."); return None
        parent_dict = parents[0]
        if self.debug_mode: print("\n >>> Generating individual using [ e2 ]...")
        child_dict = self._generate_individual_modules(parent_dict=parent_dict, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed code dictionary: \n", child_dict)
        return child_dict

    def m1(self, parent: dict) -> dict | None:
        """Mutation variation 1 using separate calls or fixed code."""
        if not parent: print("Error [m1]: No parent provided."); return None
        parent_dict = parent
        if self.debug_mode: print("\n >>> Generating individual using [ m1 ]...")
        child_dict = self._generate_individual_modules(parent_dict=parent_dict, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed code dictionary: \n", child_dict)
        return child_dict

    def m2(self, parent: dict) -> dict | None:
        """Mutation variation 2 using separate calls or fixed code."""
        if not parent: print("Error [m2]: No parent provided."); return None
        parent_dict = parent
        if self.debug_mode: print("\n >>> Generating individual using [ m2 ]...")
        child_dict = self._generate_individual_modules(parent_dict=parent_dict, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed code dictionary: \n", child_dict)
        return child_dict

    def m3(self, parent: dict) -> dict | None:
        """Mutation variation 3 using separate calls or fixed code."""
        if not parent: print("Error [m3]: No parent provided."); return None
        parent_dict = parent
        if self.debug_mode: print("\n >>> Generating individual using [ m3 ]...")
        child_dict = self._generate_individual_modules(parent_dict=parent_dict, fixed_modules=self.fixed_modules_code)
        if self.debug_mode and child_dict: print("\n >>> check designed code dictionary: \n", child_dict)
        return child_dict