from collections.abc import Iterator, Sequence
import multiprocessing
import os
import typing
from typing import Protocol, SupportsIndex, TypeVar
from pathlib import Path

import jax
import jax.numpy as jnp
import lerobot.common.datasets.lerobot_dataset as lerobot_dataset
import numpy as np
import torch
from openpi.training.dvrk_dataset import EpisodicDatasetDvrkGeneric
from torch.utils.data import Sampler
import openpi.models.model as _model
import openpi.training.config as _config
import openpi.transforms as _transforms
import random

T_co = TypeVar("T_co", covariant=True)


class Dataset(Protocol[T_co]):
    """Interface for a dataset with random access."""

    def __getitem__(self, index: SupportsIndex) -> T_co:
        raise NotImplementedError("Subclasses of Dataset should implement __getitem__.")

    def __len__(self) -> int:
        raise NotImplementedError("Subclasses of Dataset should implement __len__.")


class DataLoader(Protocol[T_co]):
    """Interface for a data loader."""

    def data_config(self) -> _config.DataConfig:
        """Get the data config for this data loader."""
        raise NotImplementedError("Subclasses of DataLoader should implement data_config.")

    def __iter__(self) -> Iterator[T_co]:
        raise NotImplementedError("Subclasses of DataLoader should implement __iter__.")


class TransformedDataset(Dataset[T_co]):
    def __init__(self, dataset: Dataset, transforms: Sequence[_transforms.DataTransformFn]):
        self._dataset = dataset
        self._transform = _transforms.compose(transforms)

    def __getitem__(self, index: SupportsIndex) -> T_co:
        return self._transform(self._dataset[index])

    def __len__(self) -> int:
        return len(self._dataset)


class FakeDataset(Dataset):
    def __init__(self, model_config: _model.BaseModelConfig, num_samples: int):
        self._num_samples = num_samples
        self._observation_spec, self._action_spec = model_config.inputs_spec()

    def __getitem__(self, index: SupportsIndex) -> dict:
        rng = jax.random.key(index.__index__())

        def make_from_spec(spec: jax.ShapeDtypeStruct):
            nonlocal rng
            rng, data_rng = jax.random.split(rng)
            # Remove the batch dimension.
            shape = spec.shape[1:]
            if spec.dtype == jnp.float32:
                return jax.random.uniform(data_rng, shape=shape, minval=-1.0, maxval=1.0)
            if spec.dtype == jnp.int32:
                return jax.random.randint(data_rng, shape=shape, minval=0, maxval=2048)
            return jnp.zeros(shape=shape, dtype=spec.dtype)

        observation = jax.tree.map(make_from_spec, self._observation_spec)
        action = jax.tree.map(make_from_spec, self._action_spec)

        return {
            **observation.to_dict(),
            "actions": action,
        }

    def __len__(self) -> int:
        return self._num_samples


def create_dataset(data_config: _config.DataConfig, model_config: _model.BaseModelConfig, skip_images: bool = False, is_eval: bool = False) -> Dataset:
    """Create a dataset for training."""
    repo_id = data_config.repo_id
    if repo_id is None:
        raise ValueError("Repo ID is not set. Cannot create dataset.")
    if repo_id == "fake":
        return FakeDataset(model_config, num_samples=1024)

    if is_eval:
        dataset_dirs = data_config.eval_dataset_dirs
        assert dataset_dirs is not None, "Eval dataset dirs are not set. Cannot create dataset."
        assert len(dataset_dirs) > 0, "Eval dataset dirs are empty. Cannot create dataset."
    else:
        dataset_dirs = data_config.train_dataset_dirs
        assert dataset_dirs is not None, "Dataset dirs are not set. Cannot create dataset."
        assert len(dataset_dirs) > 0, "Dataset dirs are empty. Cannot create dataset."

    dataset = EpisodicDatasetDvrkGeneric(
        robot_base_dir_list=dataset_dirs,
        action_horizon=50, # Use action horizon from args
        cutting_action_pad_size=10, # Keep default or make arg
        # No transforms needed usually, dataset handles internal processing
        skip_images=skip_images,
        is_eval=is_eval,
        targeting_strategy=data_config.targeting_strategy,
    )

    return dataset


def transform_dataset(dataset: Dataset, data_config: _config.DataConfig, *, skip_norm_stats: bool = False) -> Dataset:
    """Transform the dataset by applying the data transforms."""
    norm_stats = {}
    if data_config.repo_id != "fake" and not skip_norm_stats:
        if data_config.norm_stats is None:
            raise ValueError(
                "Normalization stats not found. "
                "Make sure to run `scripts/compute_norm_stats.py --config-name=<your-config>`."
            )
        norm_stats = data_config.norm_stats

    return TransformedDataset(
        dataset,
        [
            *data_config.repack_transforms.inputs,
            *data_config.data_transforms.inputs,
            _transforms.Normalize(norm_stats, use_quantiles=data_config.use_quantile_norm),
            *data_config.model_transforms.inputs,
        ],
    )


def create_data_loader(
    config: _config.TrainConfig,
    *,
    sharding: jax.sharding.Sharding | None = None,
    skip_norm_stats: bool = False,
    shuffle: bool = False,
    num_batches: int | None = None,
    num_workers: int = 0,
    is_eval: bool = False,
) -> DataLoader[tuple[_model.Observation, _model.Actions]]:
    """Create a data loader for training.

    Args:
        config: The training configuration.
        sharding: The sharding to use for the data loader. If None, the data loader will
            use a single device sharding.
        skip_norm_stats: Whether to skip data normalization.
        shuffle: Whether to shuffle the data.
        num_batches: Determines the number of batches to return. If the number exceeds the
            number of batches in the dataset, the data loader will loop over the dataset.
            If not provided, will iterate over the dataset indefinitely.
        num_workers: The number of worker processes to use. If zero, the data loader will
            execute in the main process.
        is_eval: Whether to create an evaluation data loader.
    """
    data_config = config.data.create(config.assets_dirs, config.model)

    dataset = create_dataset(data_config, config.model, is_eval=is_eval)
    dataset = transform_dataset(dataset, data_config, skip_norm_stats=skip_norm_stats)

    base_dataset = dataset._dataset if isinstance(dataset, TransformedDataset) else dataset

    if config.balance_data:
        print("using task-balanced sampler")
        # Use task-balanced sampler instead of random shuffling

        # Check for tasks with no samples
        empty_tasks = [k for k, v in base_dataset.task_to_indices.items() if len(v) == 0]
        if empty_tasks:
            raise ValueError(f"Tasks with no samples: {empty_tasks}")
        
        samples_per_task = config.batch_size // len(base_dataset.task_to_indices.keys())
        print("Attempting to sample", samples_per_task, "samples per task")

        # (Optional) Check if any task has fewer samples than desired
        too_small = [k for k, v in base_dataset.task_to_indices.items() if len(v) < samples_per_task]
        if too_small:
            print(f" Warning: Tasks with < {samples_per_task} samples: {too_small}")
            # Optionally adjust samples_per_task to the minimum
            samples_per_task = min(len(v) for v in base_dataset.task_to_indices.values())

        if config.sampler == "stratified":
            sampler = StratifiedBatchSampler(
                task_to_indices=base_dataset.task_to_indices,
                batch_size=config.batch_size,
                # batch_size=num_tasks * samples_per_task,
                samples_per_task=samples_per_task,
                seed=config.seed,
            )
        else:
            sampler = NeedleFocusedBatchSampler(
                task_to_indices=base_dataset.task_to_indices,
                instruction_to_idx=base_dataset.instruction_to_idx,
                batch_size=config.batch_size,
                seed=config.seed,
            )

        data_loader = TorchDataLoader(
            dataset,
            local_batch_size=config.batch_size // jax.process_count(),
            sharding=sharding,
            #num_batches=num_batches,
            num_workers=num_workers,
            seed=config.seed,
            batch_sampler=sampler,  # Use custom sampler
        )

    else:
        data_loader = TorchDataLoader(
            dataset,
            local_batch_size=config.batch_size // jax.process_count(),
            sharding=sharding,
            shuffle=shuffle,
            num_batches=num_batches,
            num_workers=num_workers,
            seed=config.seed,
        )

    class DataLoaderImpl(DataLoader):
        def __init__(self, data_config: _config.DataConfig, data_loader: TorchDataLoader):
            self._data_config = data_config
            self._data_loader = data_loader

        def data_config(self) -> _config.DataConfig:
            return self._data_config

        def __iter__(self):
            for batch in self._data_loader:
                yield _model.Observation.from_dict(batch), batch["actions"]

    return DataLoaderImpl(data_config, data_loader)


class TorchDataLoader:
    def __init__(
        self,
        dataset,
        local_batch_size: int,
        *,
        batch_sampler: Sampler | None = None,
        sharding: jax.sharding.Sharding | None = None,
        shuffle: bool = False,
        num_batches: int | None = None,
        num_workers: int = 0,
        seed: int = 0,
    ):
        """Create a PyTorch data loader.

        Args:
            dataset: The dataset to load.
            local_batch_size: The local batch size for each process.
            sharding: The sharding to use for the data loader.
            shuffle: Whether to shuffle the data.
            num_batches: If provided, determines the number of returned batches. If the
                number is larger than the number of batches in the dataset, the data loader
                will loop over the dataset. If not provided, will iterate over the dataset
                indefinitely.
            num_workers: The number of worker processes to use. If zero, the data loader will
                execute in the main process.
            seed: The seed to use for shuffling the data.
        """
        if jax.process_count() > 1:
            raise NotImplementedError("Data loading with multiple processes is not supported.")

        if len(dataset) < local_batch_size:
            raise ValueError(f"Local batch size ({local_batch_size}) is larger than the dataset size ({len(dataset)}).")

        if sharding is None:
            sharding = jax.sharding.SingleDeviceSharding(jax.devices()[0])
        self._sharding = sharding
        self._num_batches = num_batches

        mp_context = None
        if num_workers > 0:
            mp_context = multiprocessing.get_context("spawn")

        generator = torch.Generator()
        generator.manual_seed(seed)
        if batch_sampler:
            self._data_loader = torch.utils.data.DataLoader(
                typing.cast(torch.utils.data.Dataset, dataset),
                num_workers=num_workers,
                multiprocessing_context=mp_context,
                persistent_workers=num_workers > 0,
                collate_fn=_collate_fn,
                worker_init_fn=_worker_init_fn,
                generator=generator,
                batch_sampler=batch_sampler
            )
            
        else:
            self._data_loader = torch.utils.data.DataLoader(
                typing.cast(torch.utils.data.Dataset, dataset),
                batch_size=local_batch_size,
                shuffle=shuffle,
                num_workers=num_workers,
                multiprocessing_context=mp_context,
                persistent_workers=num_workers > 0,
                collate_fn=_collate_fn,
                worker_init_fn=_worker_init_fn,
                drop_last=True,
                generator=generator,
                batch_sampler=batch_sampler
            )

    @property
    def torch_loader(self) -> torch.utils.data.DataLoader:
        return self._data_loader

    def __iter__(self):
        num_items = 0
        while True:
            data_iter = iter(self._data_loader)
            while True:
                if self._num_batches is not None and num_items >= self._num_batches:
                    return
                try:
                    batch = next(data_iter)
                except StopIteration:
                    break  # We've exhausted the dataset. Create a new iterator and start over.
                num_items += 1
                yield jax.tree.map(lambda x: jax.make_array_from_process_local_data(self._sharding, x), batch)


class StratifiedBatchSampler(Sampler):
    def __init__(self, task_to_indices, batch_size, samples_per_task=3, seed=0):
        self.task_to_indices = task_to_indices
        self.task_ids = list(task_to_indices.keys())
        self.batch_size = batch_size
        self.samples_per_task = samples_per_task
        self.seed = seed
        self.all_indices = [idx for indices in task_to_indices.values() for idx in indices]

    def __iter__(self):
        random.seed(self.seed)
        task_pools = {
            task: random.sample(indices, len(indices))
            for task, indices in self.task_to_indices.items()
        }

        while True:
            batch = []
            # Sample at least `samples_per_task` from each task
            for task in self.task_ids:
                pool = task_pools[task]
                if len(pool) < self.samples_per_task:
                    task_pools[task] = random.sample(self.all_indices, len(self.all_indices))
                    pool = task_pools[task]
                    if len(pool) < self.samples_per_task:
                        raise ValueError(f"Not enough samples in task {task} to satisfy `samples_per_task`.")
                batch.extend([pool.pop() for _ in range(self.samples_per_task)])

            # Randomly sample the remaining samples to fill the batch
            remaining_samples = self.batch_size - len(batch)
            if remaining_samples > 0:
                batch.extend(random.sample(self.all_indices, remaining_samples))

            yield batch

    def __len__(self):
        min_size = min(len(indices) for indices in self.task_to_indices.values())
        return min_size // self.samples_per_task


class NeedleFocusedBatchSampler(Sampler):
    def __init__(self, task_to_indices, instruction_to_idx, batch_size, seed=0):
        self.task_to_indices = task_to_indices
        self.instruction_to_idx = instruction_to_idx
        self.batch_size = batch_size
        self.seed = seed

        self.needle_indices = []
        self.other_indices = []
        self.needle_tasks = []
        self.other_tasks = []

        for task, index in self.instruction_to_idx.items():
            if "needle" in task.lower():
                self.needle_indices.extend(self.task_to_indices[index])
                self.needle_tasks.append(task)
            else:
                self.other_indices.extend(self.task_to_indices[index])
                self.other_tasks.append(task)
        
        if not self.needle_indices:
            raise ValueError("StratifiedBatchSampler requires at least one 'needle' task, but none were found.")
        if not self.other_indices:
             raise ValueError("StratifiedBatchSampler requires at least one non-'needle' task, but none were found.")

        print(f"Stratified Sampler: Found {len(self.needle_tasks)} 'needle' tasks ({len(self.needle_indices)} samples) and {len(self.other_tasks)} 'other' tasks ({len(self.other_indices)} samples).")
        print(f"Needle tasks: {self.needle_tasks}")
        print(f"Other tasks: {self.other_tasks}")

        self.needle_samples_per_batch = self.batch_size // 2
        self.other_samples_per_batch = self.batch_size - self.needle_samples_per_batch

        if len(self.needle_indices) < self.needle_samples_per_batch:
            raise ValueError(
                f"Not enough 'needle' samples ({len(self.needle_indices)}) to fill 50% of the batch "
                f"({self.needle_samples_per_batch})."
            )
        if len(self.other_indices) < self.other_samples_per_batch:
             raise ValueError(
                f"Not enough 'other' task samples ({len(self.other_indices)}) to fill 50% of the batch "
                f"({self.other_samples_per_batch})."
            )


    def __iter__(self):
        random.seed(self.seed)
        
        # Initialize shuffled pools
        needle_pool = random.sample(self.needle_indices, len(self.needle_indices))
        other_pool = random.sample(self.other_indices, len(self.other_indices))

        while True:
            batch = []
            
            # Replenish pools if needed
            if len(needle_pool) < self.needle_samples_per_batch:
                # Check if we have enough samples *at all* before reshuffling
                if len(self.needle_indices) < self.needle_samples_per_batch:
                     raise ValueError(f"Warning: Not enough total 'needle' samples ({len(self.needle_indices)}) to continue providing {self.needle_samples_per_batch} per batch. Stopping iteration.")
                needle_pool = random.sample(self.needle_indices, len(self.needle_indices))

            if len(other_pool) < self.other_samples_per_batch:
                if len(self.other_indices) < self.other_samples_per_batch:
                    raise ValueError(f"Warning: Not enough total 'other' samples ({len(self.other_indices)}) to continue providing {self.other_samples_per_batch} per batch. Stopping iteration.")
                other_pool = random.sample(self.other_indices, len(self.other_indices))

            # Sample from needle pool
            batch.extend([needle_pool.pop() for _ in range(self.needle_samples_per_batch)])

            # Sample from other pool
            batch.extend([other_pool.pop() for _ in range(self.other_samples_per_batch)])
            
            # Shuffle the final batch
            random.shuffle(batch)

            yield batch

    def __len__(self):
        # The number of batches is limited by the group that runs out first
        num_needle_batches = len(self.needle_indices) // self.needle_samples_per_batch if self.needle_samples_per_batch > 0 else float('inf')
        num_other_batches = len(self.other_indices) // self.other_samples_per_batch if self.other_samples_per_batch > 0 else float('inf')
        
        # If either is infinite (because 0 samples required), length is determined by the other
        if num_needle_batches == float('inf'):
            return num_other_batches if num_other_batches != float('inf') else 0 # Avoid inf len
        if num_other_batches == float('inf'):
            return num_needle_batches # num_needle_batches cannot also be inf here
            
        return min(num_needle_batches, num_other_batches)


def _collate_fn(items):
    """Collate the batch elements into batched numpy arrays."""
    # Make sure to convert to numpy arrays before stacking since some of the incoming elements
    # may be JAX arrays.
    return jax.tree.map(lambda *x: np.stack(np.asarray(x), axis=0), *items)


def _worker_init_fn(worker_id: int) -> None:
    """Tell JAX inside the worker process not to preallocate the GPU memory."""
    # NOTE: This is called after jax is imported inside the worker process. This
    # means that this approach will not work for selecting the backend.
    os.environ["XLA_PYTHON_CLIENT_PREALLOCATE"] = "false"
    os.environ["XLA_PYTHON_CLIENT_ALLOCATOR"] = "platform"
