import copy
import itertools
from abc import ABC
from typing import Any, List

import numpy as np
import pandas as pd
import tqdm
import random

from rules import Operator, Rule, Condition, TwoWeightKnapsackRule, IntegerKnapsackRule

def swap_high_rule(current_knapsack: TwoWeightKnapsackRule, rule_candidates: List[Rule]) -> TwoWeightKnapsackRule:
    """
    Modifies the current knapsack by removing a high weight rule at random and adding a different high weight rule.
    """
    if len(current_knapsack.high_weight_rules) == 0: 
        return current_knapsack
    
    rule_to_add = copy.deepcopy(rule_candidates[np.random.randint(0, len(rule_candidates))])
    
    high_weight_rules_copy = current_knapsack.high_weight_rules.copy()
    high_rule = high_weight_rules_copy[np.random.randint(0, len(high_weight_rules_copy))]
    
    high_weight_rules_copy.remove(high_rule)
    high_weight_rules_copy.append(rule_to_add)
    
    return TwoWeightKnapsackRule(
        low_weight_rules=current_knapsack.low_weight_rules,
        high_weight_rules=high_weight_rules_copy,
        name='Removed %s from High, Added %s' % (high_rule, rule_to_add),
    )

def swap_low_rule(current_knapsack: TwoWeightKnapsackRule, rule_candidates: List[Rule]) -> TwoWeightKnapsackRule:
    """
    Modifies the current knapsack by removing a low weight rule at random and adding a different low weight rule.
    """
    rule_to_add = copy.deepcopy(rule_candidates[np.random.randint(0, len(rule_candidates))])
    
    low_weight_rules_copy = current_knapsack.low_weight_rules.copy()
    low_rule = low_weight_rules_copy[np.random.randint(0, len(low_weight_rules_copy))]
    
    low_weight_rules_copy.remove(low_rule)
    low_weight_rules_copy.append(rule_to_add)
    
    return TwoWeightKnapsackRule(
        low_weight_rules=low_weight_rules_copy,
        high_weight_rules=current_knapsack.high_weight_rules,
        name='Removed %s from Low, Added %s' % (low_rule, rule_to_add),
    )

def move_low_to_high(current_knapsack): 
    if len(current_knapsack.high_weight_rules) >= len(current_knapsack.low_weight_rules):
        return move_high_to_low(current_knapsack) 
    
    high_weight_rules_copy = copy.deepcopy(current_knapsack.high_weight_rules)
    low_weight_rules_copy = copy.deepcopy(current_knapsack.low_weight_rules)

    low_rule = low_weight_rules_copy[np.random.randint(0, len(low_weight_rules_copy))]
    low_weight_rules_copy.remove(low_rule)

    high_weight_rules_copy.append(copy.deepcopy(low_rule))

    return TwoWeightKnapsackRule(
        high_weight_rules=high_weight_rules_copy,
        low_weight_rules=low_weight_rules_copy,
        name='Moved %s from Low to High' % (low_rule),
    )

def move_high_to_low(current_knapsack): 
    if len(current_knapsack.high_weight_rules) <= 1: 
        return current_knapsack
    
    high_weight_rules_copy = copy.deepcopy(current_knapsack.high_weight_rules)
    low_weight_rules_copy = copy.deepcopy(current_knapsack.low_weight_rules)

    high_rule = high_weight_rules_copy[np.random.randint(0, len(high_weight_rules_copy))]
    high_weight_rules_copy.remove(high_rule)

    low_weight_rules_copy.append(copy.deepcopy(high_rule))

    return TwoWeightKnapsackRule(
        high_weight_rules=high_weight_rules_copy,
        low_weight_rules=low_weight_rules_copy,
        name='Moved %s from High to Low' % (high_rule),
    )
