"""
Configuration file for ABBR value experiments.
Modify these parameters to change experiment settings.
"""

from dataclasses import dataclass
from typing import Type
import sys
import os

# Add parent directory to path to import datasets
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from datasets import Dataset, Recidivism, Diabetes, FICO, Schizo, Adults, Readmission

@dataclass
class ExperimentConfig:
    """Configuration for ABBR value experiments"""
    
    # Dataset selection
    dataset_class: Type[Dataset] = FICO
    
    # Confidence threshold for binary predictions (e.g., 0.9 = top 10% most confident predictions)
    confidence_threshold: float = 0.9
    
    # Rule generation parameters
    num_rules_to_generate: int = 10000  # Total number of rules to attempt generating
    max_conditions_per_rule: int = 3    # Maximum number of conditions in each rule
    min_rule_support: float = 0.1       # Minimum support (coverage) required for a rule
    max_valid_rules: int = 2500         # Stop generating after this many valid rules
    
    # Rule selection parameters
    top_k_rules: int = 1                # Number of top rules to select for each metric (for multi-seed experiment, use 1)
    
    # Multi-seed experiment parameters
    num_seeds: int = 100                # Number of random seeds to test
    base_seed: int = 1                  # Starting seed (will use base_seed to base_seed + num_seeds - 1)
    
    # Output settings
    save_detailed_results: bool = True  # Save detailed results for each seed
    save_summary_results: bool = True   # Save summary statistics
    output_dir: str = "results"         # Directory to save results
    
    def get_output_prefix(self) -> str:
        """Generate output file prefix based on configuration"""
        return f"abbr_experiment_{self.dataset_class.__name__}_thresh{self.confidence_threshold}_support{self.min_rule_support}_seeds{self.num_seeds}"

# Available dataset classes for easy switching
AVAILABLE_DATASETS = {
    'recidivism': Recidivism,
    'diabetes': Diabetes,
    'fico': FICO,
    'schizo': Schizo,
    'adults': Adults,
    'readmission': Readmission,
} 