"""
Easy-to-use script for running ABBR value experiments.

Usage examples:
    # Run FICO experiment with default settings
    python run_abbr_experiment.py --dataset fico
    
    # Run with custom parameters
    python run_abbr_experiment.py --dataset recidivism --seeds 50 --threshold 0.8 --support 0.05
    
    # Run multiple datasets
    python run_abbr_experiment.py --dataset fico diabetes adults
"""

import argparse
import os

# Change to parent directory so data paths work correctly
script_dir = os.path.dirname(os.path.abspath(__file__))
parent_dir = os.path.dirname(script_dir)
os.chdir(parent_dir)

from abbr_experiment_config import ExperimentConfig, AVAILABLE_DATASETS
from abbr_multi_seed_experiment import run_multi_seed_experiment, print_summary_results, save_results

def main():
    parser = argparse.ArgumentParser(
        description="Run ABBR value experiment to compare ABBR vs consistency-based rule selection",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    
    parser.add_argument(
        '--dataset', 
        type=str, 
        nargs='+',
        choices=list(AVAILABLE_DATASETS.keys()),
        default=['fico'],
        help='Dataset(s) to run experiment on'
    )
    
    parser.add_argument(
        '--seeds', 
        type=int, 
        default=100,
        help='Number of random seeds to test'
    )
    
    parser.add_argument(
        '--threshold', 
        type=float, 
        default=0.9,
        help='Confidence threshold for binary predictions (e.g., 0.9 = top 10%% most confident)'
    )
    
    parser.add_argument(
        '--support', 
        type=float, 
        default=0.1,
        help='Minimum support (coverage) required for a rule'
    )
    
    parser.add_argument(
        '--max-conditions', 
        type=int, 
        default=3,
        help='Maximum number of conditions per rule'
    )
    
    parser.add_argument(
        '--max-rules', 
        type=int, 
        default=2500,
        help='Maximum number of valid rules to generate per seed'
    )
    
    parser.add_argument(
        '--num-attempts', 
        type=int, 
        default=10000,
        help='Total number of rules to attempt generating per seed'
    )
    
    parser.add_argument(
        '--output-dir', 
        type=str, 
        default='results',
        help='Directory to save results'
    )
    
    parser.add_argument(
        '--no-save', 
        action='store_true',
        help='Do not save results to files'
    )
    
    args = parser.parse_args()
    
    # Run experiment for each dataset
    for dataset_name in args.dataset:
        print(f"\n{'='*100}")
        print(f"RUNNING EXPERIMENT FOR {dataset_name.upper()}")
        print(f"{'='*100}")
        
        dataset_class = AVAILABLE_DATASETS[dataset_name]
        
        config = ExperimentConfig(
            dataset_class=dataset_class,
            confidence_threshold=args.threshold,
            num_rules_to_generate=args.num_attempts,
            max_conditions_per_rule=args.max_conditions,
            min_rule_support=args.support,
            max_valid_rules=args.max_rules,
            num_seeds=args.seeds,
            save_detailed_results=not args.no_save,
            save_summary_results=not args.no_save,
            output_dir=os.path.join('abbr_value', args.output_dir)
        )
        
        try:
            results_df, summary_stats = run_multi_seed_experiment(config)
            print_summary_results(summary_stats)
            
            if not args.no_save:
                save_results(results_df, summary_stats, config)
                
        except Exception as e:
            print(f"ERROR running experiment for {dataset_name}: {e}")
            continue
    
    print(f"\n{'='*100}")
    print(f"ALL EXPERIMENTS COMPLETED")
    print(f"{'='*100}")

if __name__ == "__main__":
    main() 