# # The Value of ABBR as a Metric 

# This notebook explains why the Average Black-Box Ranking (ABBR) is a suitable metric for evaluating how well a rule or rule list explains the confident predictions of a black-box model. In particular, we show that ABBR is a better metric than _consistency_, which is a more traditional metric for evaluating rules against 0/1 predictions, ignoring the underlying confidence of the predictions. 

# To show the value of ABBR, we show that consistency is not a very robust metric when the confidence threshold is high. Here, we define robustness as the abilty to generalize to unseen data, i.e. from training set to testing set. We set up our experiment as follows: 

# 1. Given a dataseet, we divide it into a 70% training set and a 30% testing set. 
# 2. We train a black-box model (e.g., a random forest) on the training set. 
# 3. We generate predicted probabilities on both the training set and testing set. We normalize the probabilities to be quantiles (i.e., rankings), so the data point with the highest probability is assigned a value of 1, and the data point with the lowest probability is assigned a prediction of 0. 
# 4. We generate random rules of up to 3 conditions on the training set with support at least 10\% of the dataset. For each rule, we calculate the ABBR of the rule on the training set, as well as the consistency of the rule on the training set, with a cutoff of say $0.9$. 
# 5. Out of these rules, select the top 10 rules based on ABBR. Select the top 10 rules based on consistency with the 0.9 threshold.
# 6. Evaluate the consistency of both sets of top 10 rules on the testing set for consistency with the 0.9 threshold. 

from datasets import Recidivism, Diabetes, FICO, Schizo, Adults, Readmission
from rules import Rule, Condition, Operator
import pandas as pd
import numpy as np
import random
from itertools import combinations
from IPython import embed

# ============================================================================
# EXPERIMENT PARAMETERS - Modify these to change experiment settings
# ============================================================================

# Confidence threshold for binary predictions (e.g., 0.9 = top 10% most confident predictions)
CONFIDENCE_THRESHOLD = 0.9

# Rule generation parameters
NUM_RULES_TO_GENERATE = 10000  # Total number of rules to attempt generating
MAX_CONDITIONS_PER_RULE = 3   # Maximum number of conditions in each rule
MIN_RULE_SUPPORT = 0.1       # Minimum support (coverage) required for a rule
MAX_VALID_RULES = 2500        # Stop generating after this many valid rules

# Rule selection parameters
TOP_K_RULES = 1              # Number of top rules to select for each metric

# Random seed for reproducibility
RANDOM_SEED = 1

# ============================================================================

# Set random seed for reproducibility
random.seed(RANDOM_SEED)
np.random.seed(RANDOM_SEED)

recidivism = FICO()

X_train = recidivism.get_X_train()
X_test = recidivism.get_X_test()
y_train_quantile = recidivism.get_y_train_quantile()
y_test_quantile = recidivism.get_y_test_quantile()

y_train_preds_90 = recidivism.get_y_train_preds(CONFIDENCE_THRESHOLD)
y_test_preds_90 = recidivism.get_y_test_preds(CONFIDENCE_THRESHOLD)

print("Dataset loaded:")
print(f"Training set size: {len(X_train)}")
print(f"Test set size: {len(X_test)}")
print(f"Number of features: {len(X_train.columns)}")
print(f"Confidence threshold: {CONFIDENCE_THRESHOLD}")
print(f"Minimum rule support: {MIN_RULE_SUPPORT}")
print(f"Maximum conditions per rule: {MAX_CONDITIONS_PER_RULE}")

# Debug: Show feature types
print(f"\nFeature type analysis:")
feature_types = {
    'object': 0,
    'bool': 0, 
    'binary_numeric': 0,
    'numeric': 0,
    'other': 0
}

for feature in X_train.columns:
    if X_train[feature].dtype == 'object':
        feature_types['object'] += 1
    elif X_train[feature].dtype == 'bool':
        feature_types['bool'] += 1
    elif X_train[feature].nunique() <= 2:
        feature_types['binary_numeric'] += 1
    elif np.issubdtype(X_train[feature].dtype, np.number):
        feature_types['numeric'] += 1
    else:
        feature_types['other'] += 1

for ftype, count in feature_types.items():
    if count > 0:
        print(f"  {ftype}: {count} features")
        
print(f"  Sample feature dtypes: {dict(list(X_train.dtypes.value_counts().head().items()))}")


# Step 4: Generate random rules with up to 3 conditions and at least 10% support
def generate_random_rules(X, y_quantile, num_rules=NUM_RULES_TO_GENERATE, max_conditions=MAX_CONDITIONS_PER_RULE, min_support=MIN_RULE_SUPPORT):
    """Generate random rules with up to max_conditions and at least min_support"""
    rules = []
    features = X.columns.tolist()
    
    for _ in range(num_rules):
        # Randomly choose number of conditions (1 to max_conditions)
        num_conditions = random.randint(1, max_conditions)
        
        # Randomly select features without replacement
        selected_features = random.sample(features, min(num_conditions, len(features)))
        
        conditions = []
        for feature in selected_features:
            # More robust detection of categorical/binary features
            is_categorical = (
                X[feature].dtype == 'object' or  # String/object type
                X[feature].dtype == 'bool' or    # Boolean type
                X[feature].nunique() <= 2 or     # Binary (including with NaN)
                str(X[feature].dtype).startswith('category')  # Pandas categorical
            )
            
            if is_categorical:
                # Categorical or binary feature - use equality
                unique_values = X[feature].dropna().unique()  # Remove NaN values
                if len(unique_values) > 0:
                    value = random.choice(unique_values)
                    operator = Operator.EQUAL
                else:
                    continue  # Skip features with no valid values
            else:
                # Numerical feature - use threshold
                try:
                    # Choose threshold based on quantiles to get reasonable splits
                    quantile = random.uniform(0.1, 0.9)
                    value = np.quantile(X[feature].dropna(), quantile)  # Remove NaN values
                    operator = random.choice([Operator.LESS, Operator.GREATER])
                except (TypeError, ValueError) as e:
                    # If quantile calculation fails, treat as categorical
                    unique_values = X[feature].dropna().unique()
                    if len(unique_values) > 0:
                        value = random.choice(unique_values)
                        operator = Operator.EQUAL
                    else:
                        continue  # Skip features with no valid values
            
            conditions.append(Condition(feature, operator, value))
        
        # Skip if no valid conditions were created
        if len(conditions) == 0:
            continue
            
        rule = Rule(conditions)
        
        # Check if rule has sufficient support
        mask = rule.get_mask(X)
        support = np.mean(mask)
        
        if support >= min_support:
            rules.append(rule)
            
        # Stop if we have enough valid rules
        if len(rules) >= MAX_VALID_RULES:
            break
    
    return rules

print(f"\nGenerating random rules (up to {MAX_VALID_RULES} valid rules)...")
random_rules = generate_random_rules(X_train, y_train_quantile)
print(f"Generated {len(random_rules)} valid rules with support >= {MIN_RULE_SUPPORT}")


# Step 5: Calculate ABBR and consistency for each rule on training set
def calculate_metrics(rules, X, y_quantile, y_binary):
    """Calculate ABBR and consistency metrics for each rule"""
    rule_metrics = []
    
    for rule in rules:
        mask = rule.get_mask(X)
        support = np.mean(mask)
        
        if support > 0:  # Avoid division by zero
            # ABBR: Average quantile of covered points
            abbr = np.mean(y_quantile[mask])
            
            # Consistency: Proportion of covered points that are positive (with threshold)
            consistency = np.mean(y_binary[mask])
            
            rule_metrics.append({
                'rule': rule,
                'support': support,
                'abbr': abbr,
                'consistency': consistency
            })
    
    return pd.DataFrame(rule_metrics)

print("\nCalculating ABBR and consistency metrics...")
metrics_df = calculate_metrics(random_rules, X_train, y_train_quantile, y_train_preds_90)
print(f"Calculated metrics for {len(metrics_df)} rules")

# Print summary statistics
print(f"\nMetrics summary:")
print(f"ABBR: mean={metrics_df['abbr'].mean():.3f}, std={metrics_df['abbr'].std():.3f}")
print(f"Consistency: mean={metrics_df['consistency'].mean():.3f}, std={metrics_df['consistency'].std():.3f}")
print(f"Support: mean={metrics_df['support'].mean():.3f}, std={metrics_df['support'].std():.3f}")


# Step 6: Select top K rules based on ABBR and top K based on consistency
top_abbr_rules = metrics_df.nlargest(TOP_K_RULES, 'abbr')
top_consistency_rules = metrics_df.nlargest(TOP_K_RULES, 'consistency')

print(f"\nTop {TOP_K_RULES} rules by ABBR:")
for i, row in top_abbr_rules.iterrows():
    print(f"Rule {i}: ABBR={row['abbr']:.3f}, Consistency={row['consistency']:.3f}, Support={row['support']:.3f}")

print(f"\nTop {TOP_K_RULES} rules by Consistency:")
for i, row in top_consistency_rules.iterrows():
    print(f"Rule {i}: ABBR={row['abbr']:.3f}, Consistency={row['consistency']:.3f}, Support={row['support']:.3f}")


# Step 7: Evaluate both sets of rules on test set
def evaluate_rules_on_test(rules_df, X_test, y_test_binary):
    """Evaluate consistency of rules on test set"""
    test_consistencies = []
    
    for _, row in rules_df.iterrows():
        rule = row['rule']
        mask = rule.get_mask(X_test)
        
        if np.sum(mask) > 0:  # Rule covers some points
            test_consistency = np.mean(y_test_binary[mask])
            test_support = np.mean(mask)
        else:
            test_consistency = 0
            test_support = 0
            
        test_consistencies.append({
            'train_abbr': row['abbr'],
            'train_consistency': row['consistency'],
            'train_support': row['support'],
            'test_consistency': test_consistency,
            'test_support': test_support
        })
    
    return pd.DataFrame(test_consistencies)

print(f"\nEvaluating rules on test set...")

# Evaluate ABBR-selected rules on test set
abbr_test_results = evaluate_rules_on_test(top_abbr_rules, X_test, y_test_preds_90)
abbr_test_results['selection_method'] = 'ABBR'

# Evaluate consistency-selected rules on test set  
consistency_test_results = evaluate_rules_on_test(top_consistency_rules, X_test, y_test_preds_90)
consistency_test_results['selection_method'] = 'Consistency'

# Combine results
all_results = pd.concat([abbr_test_results, consistency_test_results], ignore_index=True)

print(f"\nResults Summary:")
print(f"ABBR-selected rules:")
print(f"  Average train consistency: {abbr_test_results['train_consistency'].mean():.3f}")
print(f"  Average test consistency: {abbr_test_results['test_consistency'].mean():.3f}")
print(f"  Average test support: {abbr_test_results['test_support'].mean():.3f}")
print(f"  Consistency drop: {abbr_test_results['train_consistency'].mean() - abbr_test_results['test_consistency'].mean():.3f}")

print(f"\nConsistency-selected rules:")
print(f"  Average train consistency: {consistency_test_results['train_consistency'].mean():.3f}")
print(f"  Average test consistency: {consistency_test_results['test_consistency'].mean():.3f}")
print(f"  Average test support: {consistency_test_results['test_support'].mean():.3f}")

print(f"  Consistency drop: {consistency_test_results['train_consistency'].mean() - consistency_test_results['test_consistency'].mean():.3f}")

# Calculate generalization gap (robustness measure)
abbr_gap = abbr_test_results['train_consistency'].mean() - abbr_test_results['test_consistency'].mean()
consistency_gap = consistency_test_results['train_consistency'].mean() - consistency_test_results['test_consistency'].mean()

print(f"\nGeneralization Analysis:")
print(f"ABBR-selected rules generalization gap: {abbr_gap:.3f}")
print(f"Consistency-selected rules generalization gap: {consistency_gap:.3f}")
print(f"ABBR shows {'better' if abbr_gap < consistency_gap else 'worse'} generalization (smaller gap is better)")

# Save results for further analysis
output_filename = f'abbr_value_experiment_results_threshold_{CONFIDENCE_THRESHOLD}_support_{MIN_RULE_SUPPORT}.csv'
all_results.to_csv(output_filename, index=False)
print(f"\nResults saved to '{output_filename}'")