"""Tests for utils.py."""

import pytest

from fmri2music import utils


@pytest.mark.parametrize(
    "input_string,expected_output",
    [
        ("sample_1.23456_7.890123", "sample_1.23_7.89"),
        ("sample_0_0", "sample_0.00_0.00"),
        ("sample_-3.45678_9.101112", "sample_-3.46_9.10"),
    ],
)
def test_normalize_slice_name_valid(input_string, expected_output):
    assert utils.normalize_slice_name(input_string) == expected_output


@pytest.mark.parametrize(
    "input_string",
    [
        "sample_1.23",
        "sample",
        "1.23_7.89",
    ],
)
def test_normalize_slice_name_invalid_format(input_string):
    with pytest.raises(ValueError):
        utils.normalize_slice_name(input_string)


def test_normalize_slice_name_invalid_value():
    with pytest.raises(ValueError):
        utils.normalize_slice_name("sample_key_start_end")


@pytest.mark.parametrize(
    "start_s,end_s,start_idx,end_idx",
    [
        (0, 1.5, 0, 1),
        (0, 1.5001, 0, 1),
        (1.6, 3, 1, 2),
        (0, 15, 0, 10),
        (0.1, 14.9, 0, 10),
        (2, 4, 1, 3),
    ],
)
def test_seconds_to_idx(start_s: float, end_s: float, start_idx: int, end_idx: int):
    assert utils.seconds_to_idx(start_s, end_s) == (start_idx, end_idx)


@pytest.mark.parametrize(
    "names, expected",
    [
        (
            ["P1_0_1", "P2_2_3", "P1_1_2", "P2_3_4"],
            [["P1_0_1", "P1_1_2"], ["P2_2_3", "P2_3_4"]],
        ),
        (
            ["P3_0_1", "P2_1_2", "P1_2_3", "P1_3_4"],
            [["P1_2_3", "P1_3_4"], ["P2_1_2"], ["P3_0_1"]],
        ),
        (
            [],
            [],
        ),
        (  # Test sorting by start time.
            ["P1_0_1", "P1_5_6", "P1_1_2", "P1_3_4"],
            [["P1_0_1", "P1_1_2", "P1_3_4", "P1_5_6"]],
        ),
    ],
)
def test_group_stim_names(names, expected):
    assert utils.group_stim_names(names) == expected
