import numpy as np

def planning_function(processed_state):
    """
    Determines optimal tasks for each agent based on the current state.
    
    Args:
        processed_state: 
            Dict of list of (n, ): dict('agent_0', 'agent_1', 'agent_2', 'agent_3', 'agent_4')
                List: Agent : (m, ) [landmark0_2_agent, landmark1_2_agent, landmark2_2_agent, landmark3_2_agent, landmark4_2_agent, other_agent_2_agent, other_agent_2_agent, other_agent_2_agent, other_agent_2_agent]

    Returns:
        dict: Optimal tasks for each agent ('No op','Landmark_0','Landmark_1','Landmark_2','Landmark_3','Landmark_4')
    """
    num_agents = len(processed_state)
    num_landmarks = 5
    
    # Calculate distances between agents and landmarks
    distances = {agent: [np.linalg.norm(landmark) for landmark in processed_state[agent][:num_landmarks]] for agent in processed_state}
    
    # Sort agents by their closest landmark
    sorted_agents = sorted(distances.keys(), key=lambda x: min(distances[x]))
    
    assigned_landmarks = set()
    llm_tasks = {}
    
    for agent in sorted_agents:
        # Find the closest unassigned landmark
        for i, dist in sorted(enumerate(distances[agent]), key=lambda x: x[1]):
            if i not in assigned_landmarks:
                llm_tasks[agent] = f"Landmark_{i}"
                assigned_landmarks.add(i)
                break
        else:
            # If all landmarks are assigned, assign 'No op'
            llm_tasks[agent] = "No op"
    
    return llm_tasks

def compute_reward(processed_state, llm_actions, actions):
    """
    Calculate rewards based on the tasks assigned and their outcomes.
    
    Args:
        processed_state: returned from function process_state(state, p, f)
        llm_actions (dict): dictionary of list of integers which means the suggest actions from llm for each agent. E.g. {"agent_0": [2,3], "agent_1": [4],...}
        actions (dict): dictionary of a integer action that actually perform by each agent. E.g. {"agent_0": 2, "agent_1": 4, ...}

        **Note: the index of this action space is [no_action, move_left, move_right, move_down, move_up]**
        
    Returns:
        reward: Dict containing rewards for each agent. For example: {'agent_0': reward1, 'agent_1', reward2, ...}
    """
    reward = {}
    num_landmarks = 5
    collision_threshold = 0.3
    
    for agent in processed_state:
        # Initialize reward
        reward[agent] = 0
        
        # Reward for moving towards assigned landmark
        assigned_landmark = np.argmin([np.linalg.norm(landmark) for landmark in processed_state[agent][:num_landmarks]])
        distance_to_landmark = np.linalg.norm(processed_state[agent][assigned_landmark])
        reward[agent] -= distance_to_landmark  # Negative reward for distance
        
        # Penalty for collision or near-collision with other agents
        for other_agent_pos in processed_state[agent][num_landmarks:]:
            if np.linalg.norm(other_agent_pos) < collision_threshold:
                reward[agent] -= 1  # Penalty for collision
        
        # Reward for following LLM suggestions
        if actions[agent] in llm_actions[agent]:
            reward[agent] += 0.5  # Bonus for following LLM suggestion
        
    # Global reward for landmark coverage
    total_distance = sum(min(np.linalg.norm(processed_state[agent][i]) for agent in processed_state) for i in range(num_landmarks))
    global_reward = -total_distance / num_landmarks
    
    # Add global reward to each agent's reward
    for agent in reward:
        reward[agent] += global_reward
    
    return reward