import numpy as np

def planning_function(processed_state):
    """
    Determines optimal tasks for each agent based on the current state.
    
    Args:
        processed_state: 
            Dict of list of (n, ): dict('agent_0', 'agent_1', 'agent_2)
                List: Agent : (m, ) [landmark0_2_agent, landmark1_2_agent, landmark2_2_agent, other_agent_2_agent, other_agent_2_agent]

    Returns:
        dict: Optimal tasks for each agent ('No op','Landmark_0','Landmark_1','Landmark_2')
    """
    num_agents = len(processed_state)
    num_landmarks = len(processed_state['agent_0']) - (num_agents - 1)
    
    distances = {agent: [np.linalg.norm(landmark) for landmark in processed_state[agent][:num_landmarks]] for agent in processed_state}
    
    assigned_landmarks = set()
    llm_tasks = {}
    
    for _ in range(num_agents):
        min_distance = float('inf')
        best_agent = None
        best_landmark = None
        
        for agent in processed_state:
            if agent not in llm_tasks:
                for i, distance in enumerate(distances[agent]):
                    if i not in assigned_landmarks and distance < min_distance:
                        min_distance = distance
                        best_agent = agent
                        best_landmark = i
        
        if best_agent is not None:
            llm_tasks[best_agent] = f"Landmark_{best_landmark}"
            assigned_landmarks.add(best_landmark)
        else:
            break
    
    for agent in processed_state:
        if agent not in llm_tasks:
            llm_tasks[agent] = "No op"
    
    return llm_tasks

def compute_reward(processed_state, llm_actions, actions):
    """
    Calculate rewards based on the tasks assigned and their outcomes.
    
    Args:
        processed_state: returned from function process_state(state, p, f)
        llm_actions (dict): dictionary of list of integers which means the suggest actions from llm for each agent. E.g. {"agent_0": [2,3], "agent_1": [4],...}
        actions (dict): dictionary of a integer action that actually perform by each agent. E.g. {"agent_0": 2, "agent_1": 4, ...}

        **Note: the index of this action space is [no_action, move_left, move_right, move_down, move_up]**
        
    Returns:
        reward: Dict containing rewards for each agent. For example: {'agent_0': reward1, 'agent_1', reward2, ...}
    """
    reward = {}
    num_agents = len(processed_state)
    num_landmarks = len(processed_state['agent_0']) - (num_agents - 1)
    collision_threshold = 0.3
    
    for agent in processed_state:
        # Base reward
        reward[agent] = 0
        
        # Reward for moving towards assigned landmark
        assigned_landmark = np.argmin([np.linalg.norm(landmark) for landmark in processed_state[agent][:num_landmarks]])
        distance_to_landmark = np.linalg.norm(processed_state[agent][assigned_landmark])
        reward[agent] -= distance_to_landmark  # Negative reward for distance
        
        # Penalty for not following LLM suggestion
        if actions[agent] not in llm_actions[agent]:
            reward[agent] -= 0.5
        
        # Check for collisions or near-collisions
        for other_agent in processed_state[agent][num_landmarks:]:
            if np.linalg.norm(other_agent) < collision_threshold:
                reward[agent] -= 2  # Significant penalty for collision
    
    return reward