import numpy as np

def planning_function(processed_state):
    """
    Determines optimal tasks for each agent based on the current state.
    
    Args:
        processed_state: 
            Dict of list of (n, ): dict('agent_0', 'agent_1', 'agent_2)
                List: Agent : (m, ) [landmark0_2_agent, landmark1_2_agent, landmark2_2_agent, other_agent_2_agent, other_agent_2_agent]

    Returns:
        dict: Optimal tasks for each agent ('No op','Landmark_0','Landmark_1','Landmark_2')
    """
    num_agents = len(processed_state)
    num_landmarks = 3
    
    # Calculate distances between agents and landmarks
    distances = {}
    for agent_id, obs in processed_state.items():
        distances[agent_id] = [np.linalg.norm(obs[i]) for i in range(num_landmarks)]
    
    # Assign tasks based on closest unassigned landmark
    assigned_landmarks = set()
    llm_tasks = {}
    
    for _ in range(num_agents):
        min_distance = float('inf')
        best_agent = None
        best_landmark = None
        
        for agent_id, agent_distances in distances.items():
            if agent_id not in llm_tasks:
                for landmark_idx, distance in enumerate(agent_distances):
                    if landmark_idx not in assigned_landmarks and distance < min_distance:
                        min_distance = distance
                        best_agent = agent_id
                        best_landmark = landmark_idx
        
        if best_agent and best_landmark is not None:
            llm_tasks[best_agent] = f"Landmark_{best_landmark}"
            assigned_landmarks.add(best_landmark)
        else:
            # If no unassigned landmark is found, assign 'No op'
            for agent_id in processed_state.keys():
                if agent_id not in llm_tasks:
                    llm_tasks[agent_id] = "No op"
    
    return llm_tasks

def compute_reward(processed_state, llm_actions, actions):
    """
    Calculate rewards based on the tasks assigned and their outcomes.
    
    Args:
        processed_state: returned from function process_state(state, p, f)
        llm_actions (dict): dictionary of list of integers which means the suggest actions from llm for each agent. E.g. {"agent_0": [2,3], "agent_1": [4],...}
        actions (dict): dictionary of a integer action that actually perform by each agent. E.g. {"agent_0": 2, "agent_1": 4, ...}

        **Note: the index of this action space is [no_action, move_left, move_right, move_down, move_up]**
        
    Returns:
        reward: Dict containing rewards for each agent. For example: {'agent_0': reward1, 'agent_1', reward2, ...}
    """
    reward = {}
    num_landmarks = 3
    collision_threshold = 0.3
    
    for agent_id, obs in processed_state.items():
        # Reward for moving towards the assigned landmark
        landmark_distances = [np.linalg.norm(obs[i]) for i in range(num_landmarks)]
        closest_landmark_distance = min(landmark_distances)
        reward[agent_id] = -closest_landmark_distance  # Negative distance as reward
        
        # Penalty for not following LLM's suggestion
        if actions[agent_id] not in llm_actions[agent_id]:
            reward[agent_id] -= 0.5
        
        # Penalty for potential collisions
        for i in range(num_landmarks, len(obs)):
            if np.linalg.norm(obs[i]) < collision_threshold:
                reward[agent_id] -= 1.0
        
        # Bonus for reaching a landmark
        if closest_landmark_distance < 0.1:
            reward[agent_id] += 2.0
    
    return reward