function memo=f_ntc_LpSq_ADMM_dct(obs,opts,memo)
% ADMM optimization for tensor completion using Lp-Sq regularization with DCT
% Input:
%   obs: structure containing observation data and indices
%   opts: structure containing optimization parameters
%   memo: structure for storing iteration history and results

% Extract optimization parameters from opts structure
lambda=opts.para.lambda;    % Regularization parameter
rho=opts.para.rho;         % Initial penalty parameter
nu=opts.para.nu;           % Parameter update rate
p = opts.p;                % Lp norm parameter
q = opts.q;                % Sq norm parameter

% Calculate norm of ground truth for error computation
normTruth=norm(double(memo.truth(:)));

% Initialize binary mask for observed positions
B=zeros(obs.tsize); B(obs.idx)=1;

% Initialize tensor with observed values
T=zeros(obs.tsize); T(obs.idx)=obs.y;

% Initialize optimization variables
X=zeros(obs.tsize);        % Primal variable
if isfield(opts,'initL')   % Use initial guess if provided
    X = opts.initL;
end
Y=X;                       % Dual variable
Z=X;                       % Auxiliary variable

% Print optimization parameters
fprintf(['++++f_prox_t_LpSq_dct: p=' num2str(p) ', q=' num2str(q) '++++\n']);

% Main ADMM iteration loop
for iter=1:opts.MAX_ITER_OUT
    % Store previous iteration
    oldX=X;
    
    % Update auxiliary variable Z
    Z=(Y+rho*X+B.*T)./(rho+B);
    
    % Update primal variable X using proximal operator
    X_tmp=Z-Y/rho;
    X=f_prox_t_LpSq_dct(X_tmp,lambda/rho,oldX,p,q);
    
    % Record iteration history
    memo.iter=iter;
    memo.rho(iter)=rho;
    memo.eps(iter)=norm(double( X(:)-oldX(:) ))/( norm(double(oldX(:)))+eps );
    memo.err(iter)=norm(double( X(:)-memo.truth(:) ))/normTruth;
    memo.pnsr(iter)=h_Psnr(memo.truth(:),X(:));
    
    % Print progress if verbose mode is enabled
    if opts.verbose && mod(iter,memo.printerInterval)==0
        fprintf('++%d: eps=%0.2e, err=%0.2e, rho=%0.2e, PSNR=%0.2f\n', ...
            iter,memo.eps(iter),memo.err(iter),memo.rho(iter),memo.pnsr(iter));
    end
    
    % Check convergence criteria
    if ( memo.eps(iter) <opts.MAX_EPS ) && ( iter > 60 ) && memo.eps(iter)>1e-10
        fprintf('Stopped:%d: eps=%0.2e, err=%0.2e,rho=%0.2e PSNR=%0.2f\n', ...
            iter,memo.eps(iter),memo.err(iter),memo.rho(iter),memo.pnsr(iter));
        break;
    end
    
    % Update dual variable Y
    Y=Y+rho*(X-Z);
    
    % Update penalty parameter rho
    rho=min(rho*nu,opts.MAX_RHO);
end

% Store final completion result
memo.T_hat=X;