"""
Validation utilities for LapBoost package.

This module contains functions for validating inputs and parameters.
"""

import numpy as np
from typing import Tuple, Optional, Dict, Any
from sklearn.utils.validation import check_X_y, check_array


def check_inputs(X: np.ndarray, y: np.ndarray, 
                multi_output: bool = False) -> Tuple[np.ndarray, np.ndarray]:
    """
    Validate input arrays for LapBoost.
    
    Parameters
    ----------
    X : np.ndarray
        Input features
    y : np.ndarray
        Target values
    multi_output : bool, default=False
        Whether multiple outputs are allowed
        
    Returns
    -------
    tuple
        Validated (X, y) arrays
    """
    # Check X and y have proper shape
    X, y = check_X_y(
        X, y, 
        accept_sparse=True, 
        multi_output=multi_output,
        y_numeric=True
    )
    
    return X, y


def validate_parameters(params: Dict[str, Any], 
                      default_params: Dict[str, Any]) -> Dict[str, Any]:
    """
    Validate and merge parameters with defaults.
    
    Parameters
    ----------
    params : dict
        User-provided parameters
    default_params : dict
        Default parameter values
        
    Returns
    -------
    dict
        Validated parameters
    """
    if params is None:
        return default_params.copy()
        
    validated = default_params.copy()
    
    # Update with user parameters and validate
    for key, value in params.items():
        if key not in validated:
            raise ValueError(f"Unknown parameter: {key}")
            
        # Type checking for common parameters
        if key == 'k_neighbors':
            if not isinstance(value, int) or value <= 0:
                raise ValueError(f"k_neighbors must be a positive integer, got {value}")
                
        elif key in ('gamma', 'sigma', 'confidence_threshold'):
            if not isinstance(value, (int, float)) or value < 0:
                raise ValueError(f"{key} must be a non-negative number, got {value}")
                
        elif key == 'max_iter':
            if not isinstance(value, int) or value <= 0:
                raise ValueError(f"max_iter must be a positive integer, got {value}")
                
        validated[key] = value
        
    return validated
