"""
Setup script for LapBoost

This setup script allows for easy installation of the LapBoost
semi-supervised learning framework package.

Author: Anonymous 
Date: 2025-07-14
"""

from setuptools import setup, find_packages
import os

# Read the README file
def read_readme():
    """Read README.md for long description"""
    try:
        with open("README.md", "r", encoding="utf-8") as fh:
            return fh.read()
    except FileNotFoundError:
        return "LapBoost: Semi-Supervised XGBoost with LapTAO implementation"

# Read requirements from requirements.txt
def read_requirements():
    """Read requirements from requirements.txt"""
    try:
        with open("requirements.txt", "r", encoding="utf-8") as fh:
            requirements = []
            for line in fh:
                line = line.strip()
                if line and not line.startswith("#"):
                    requirements.append(line)
            return requirements
    except FileNotFoundError:
        return [
            "numpy",
            "pandas", 
            "scikit-learn>=1.0.0",
            "scipy",
            "xgboost>=1.5.0",
            "optuna",
            "matplotlib",
            "seaborn",
            "tqdm",
            "networkx"
        ]

setup(
    name="lapboost",
    version="1.0.0",
    author="Anonymous",
    author_email="Anonymous@Anonymous.com",
    description="A comprehensive semi-supervised learning framework combining XGBoost with LapTAO",
    long_description=read_readme(),
    long_description_content_type="text/markdown",
    url="https://github.com/Anonymous/LapBoost",
    project_urls={
        "Bug Tracker": "https://github.com/Anonymous/LapBoost/issues",
        "Documentation": "https://github.com/Anonymous/LapBoost/blob/main/README.md",
        "Source Code": "https://github.com/Anonymous/LapBoost",
    },
    classifiers=[
        "Development Status :: 4 - Beta",
        "Intended Audience :: Developers",
        "Intended Audience :: Science/Research",
        "License :: OSI Approved :: MIT License",
        "Operating System :: OS Independent",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3.7",
        "Programming Language :: Python :: 3.8",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Topic :: Scientific/Engineering :: Artificial Intelligence",
        "Topic :: Scientific/Engineering :: Information Analysis",
        "Topic :: Software Development :: Libraries :: Python Modules",
    ],
    keywords=[
        "machine-learning",
        "semi-supervised-learning", 
        "xgboost",
        "graph-regularization",
        "laptao",
        "pseudo-labeling",
        "ensemble-learning",
        "classification",
        "regression"
    ],
    packages=find_packages(exclude=["tests*", "docs*", "examples*"]),
    python_requires=">=3.7",
    install_requires=read_requirements(),
    extras_require={
        "dev": [
            "pytest>=6.0.0",
            "pytest-cov>=2.12.1",
            "flake8>=3.9.2",
            "black>=21.6b0",
            "isort>=5.9.1",
            "mypy>=0.910",
        ],
        "docs": [
            "sphinx>=4.0.2",
            "sphinx-rtd-theme>=0.5.2",
            "myst-parser>=0.15.1",
            "nbsphinx>=0.8.6",
        ],
        "jupyter": [
            "jupyterlab>=3.0.0",
            "ipywidgets>=7.6.3",
        ],
        "viz": [
            "matplotlib>=3.4.2",
            "seaborn>=0.11.1",
            "networkx>=2.6.2",
        ],
    },
    entry_points={
        "console_scripts": [
            "lapboost-demo=lapboost.cli:main",
        ],
    },
    package_data={
        "lapboost": ["py.typed"],
    },
    include_package_data=True,
    zip_safe=False,
    license="MIT",
)
