"""
Unit tests for LapBoost model classes.
"""

import unittest
import numpy as np
from sklearn.datasets import make_classification, make_regression
from sklearn.model_selection import train_test_split

from lapboost.core.model import LapBoostClassifier, LapBoostRegressor


class TestLapBoostClassifier(unittest.TestCase):
    """Test cases for LapBoostClassifier."""
    
    def setUp(self):
        """Set up test data."""
        # Create synthetic classification data
        X, y = make_classification(
            n_samples=100, 
            n_features=10, 
            n_classes=2, 
            random_state=42
        )
        
        # Split into labeled and unlabeled
        X_labeled, X_unlabeled, y_labeled, _ = train_test_split(
            X, y, test_size=0.8, random_state=42
        )
        
        self.X_labeled = X_labeled
        self.y_labeled = y_labeled
        self.X_unlabeled = X_unlabeled
        self.X_full = X
        
    def test_init(self):
        """Test initialization with default parameters."""
        model = LapBoostClassifier()
        self.assertEqual(model.k_neighbors, 10)
        self.assertEqual(model.gamma, 0.1)
        
    def test_fit_predict(self):
        """Test fitting and prediction."""
        model = LapBoostClassifier(random_state=42)
        model.fit(self.X_labeled, self.y_labeled, self.X_unlabeled)
        
        # Check that the model has been fitted
        self.assertTrue(hasattr(model, 'xgb_model_'))
        
        # Test prediction
        y_pred = model.predict(self.X_full)
        self.assertEqual(y_pred.shape[0], self.X_full.shape[0])
        
    def test_predict_proba(self):
        """Test probability prediction."""
        model = LapBoostClassifier(random_state=42)
        model.fit(self.X_labeled, self.y_labeled, self.X_unlabeled)
        
        # Test probability prediction
        y_proba = model.predict_proba(self.X_full)
        self.assertEqual(y_proba.shape[0], self.X_full.shape[0])
        self.assertEqual(y_proba.shape[1], len(np.unique(self.y_labeled)))
        
        # Check that probabilities sum to 1
        row_sums = np.sum(y_proba, axis=1)
        np.testing.assert_almost_equal(row_sums, np.ones_like(row_sums))


class TestLapBoostRegressor(unittest.TestCase):
    """Test cases for LapBoostRegressor."""
    
    def setUp(self):
        """Set up test data."""
        # Create synthetic regression data
        X, y = make_regression(
            n_samples=100, 
            n_features=10, 
            random_state=42
        )
        
        # Split into labeled and unlabeled
        X_labeled, X_unlabeled, y_labeled, _ = train_test_split(
            X, y, test_size=0.8, random_state=42
        )
        
        self.X_labeled = X_labeled
        self.y_labeled = y_labeled
        self.X_unlabeled = X_unlabeled
        self.X_full = X
        
    def test_init(self):
        """Test initialization with default parameters."""
        model = LapBoostRegressor()
        self.assertEqual(model.k_neighbors, 10)
        self.assertEqual(model.gamma, 0.1)
        
    def test_fit_predict(self):
        """Test fitting and prediction."""
        model = LapBoostRegressor(random_state=42)
        model.fit(self.X_labeled, self.y_labeled, self.X_unlabeled)
        
        # Check that the model has been fitted
        self.assertTrue(hasattr(model, 'xgb_model_'))
        
        # Test prediction
        y_pred = model.predict(self.X_full)
        self.assertEqual(y_pred.shape[0], self.X_full.shape[0])
        
    def test_parameters(self):
        """Test parameter validation."""
        # Test with invalid parameter
        with self.assertRaises(ValueError):
            model = LapBoostRegressor(k_neighbors=-1)
            model.fit(self.X_labeled, self.y_labeled, self.X_unlabeled)


if __name__ == '__main__':
    unittest.main()
