import argparse
import os
import torch.nn.functional as F
import pandas as pd
import torch
import torch.optim as optim
from thop import profile, clever_format
from torch.utils.data import DataLoader
from tqdm import tqdm
import time
import utils.utils as utils
from utils.model_prob import ProbabilisticModel


# train for one epoch to learn unique features
def train(net, data_loader, train_optimizer):
    net.train()
    total_loss, total_num, train_bar = 0.0, 0, tqdm(data_loader)
    epsilon = 1e-6  # Small constant for numerical stability
    vmf_strength = 0.1  # Hyperparameter to regularize the vmf loss
    sim_strength = 1  # Hyperparameter for the strength of cosine similarity regularization
    kappa_reg_strength = 0.001  # Regularization strength

    for pos_1, pos_2, target in train_bar:
        pos_1, pos_2 = pos_1.to(device), pos_2.to(device)

        mean_1, kappa = net(pos_1)
        mean_2, _ = net(pos_2)
        mean_2_detached = mean_2.detach()

        # Ensure variance is positive
        kappa = F.softplus(kappa)
        # Compute the kappa regularization loss (L2 norm)
        kappa_reg_loss = kappa_reg_strength * torch.mean(kappa ** 2)

        # Compute the vMF likelihood component
        vmf_lh = torch.exp(kappa * F.cosine_similarity(mean_1, mean_2_detached, dim=1))
        vmf_loss = vmf_strength * (-torch.log(vmf_lh + epsilon).mean())

        # Compute SimCLR loss
        out = torch.cat([mean_1, mean_2], dim=0)
        sim_matrix = torch.exp(torch.mm(out, out.t().contiguous()) / temperature)
        mask = (torch.ones_like(sim_matrix) - torch.eye(2 * batch_size, device=sim_matrix.device)).bool()
        sim_matrix = sim_matrix.masked_select(mask).view(2 * batch_size, -1)
        pos_sim = torch.exp(torch.sum(mean_1 * mean_2, dim=-1) / temperature)
        pos_sim = torch.cat([pos_sim, pos_sim], dim=0)
        similarity_loss = sim_strength * (-torch.log(pos_sim / sim_matrix.sum(dim=-1))).mean()

        # Compute the final loss
        loss = vmf_loss + similarity_loss + kappa_reg_loss

        # Backward and optimize
        train_optimizer.zero_grad()
        loss.backward()
        train_optimizer.step()


def test(net, memory_data_loader, test_data_loader):
    net.eval()
    total_top1, total_top5, total_num, feature_bank = 0.0, 0.0, 0, []
    with torch.no_grad():
        # generate feature bank of means
        for data, _, target in tqdm(memory_data_loader, desc='Feature extracting'):
            feature_mean, _, = net(data.to(device))
            feature_bank.append(feature_mean)
        # [D, N]
        feature_bank = torch.cat(feature_bank, dim=0).t().contiguous()
        # [N]
        feature_labels = torch.tensor(memory_data_loader.dataset.targets, device=feature_bank.device)
        # loop test data to predict the label by weighted knn search
        test_bar = tqdm(test_data_loader)
        for data, _, target in test_bar:
            data, target = data.to(device), target.to(device)
            feature_mean, _ = net(data)

            total_num += data.size(0)
            # compute cos similarity between each mean feature vector and feature bank ---> [B, N]
            sim_matrix = torch.mm(feature_mean, feature_bank)
            # [B, K]
            sim_weight, sim_indices = sim_matrix.topk(k=k, dim=-1)
            # [B, K]
            sim_labels = torch.gather(feature_labels.expand(data.size(0), -1), dim=-1, index=sim_indices)
            sim_weight = (sim_weight / temperature).exp()

            # counts for each class
            one_hot_label = torch.zeros(data.size(0) * k, c, device=sim_labels.device)
            # [B*K, C]
            one_hot_label = one_hot_label.scatter(dim=-1, index=sim_labels.view(-1, 1), value=1.0)
            # weighted score ---> [B, C]
            pred_scores = torch.sum(one_hot_label.view(data.size(0), -1, c) * sim_weight.unsqueeze(dim=-1), dim=1)

            pred_labels = pred_scores.argsort(dim=-1, descending=True)
            total_top1 += torch.sum((pred_labels[:, :1] == target.unsqueeze(dim=-1)).any(dim=-1).float()).item()
            total_top5 += torch.sum((pred_labels[:, :5] == target.unsqueeze(dim=-1)).any(dim=-1).float()).item()
            test_bar.set_description('Test Epoch: [{}/{}] Acc@1:{:.2f}% Acc@5:{:.2f}%'
                                     .format(epoch, epochs, total_top1 / total_num * 100, total_top5 / total_num * 100))

    return total_top1 / total_num * 100, total_top5 / total_num * 100



if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='Train Probabilistic CL with Explicit Concentration')
    parser.add_argument('--feature_dim', default=128, type=int, help='Feature dim for latent vector')
    parser.add_argument('--temperature', default=0.5, type=float, help='Temperature used in softmax')
    parser.add_argument('--k', default=200, type=int, help='Top k most similar images used to predict the label')
    parser.add_argument('--batch_size', default=512, type=int, help='Number of images in each mini-batch')
    parser.add_argument('--epochs', default=500, type=int, help='Number of sweeps over the dataset to train')
    parser.add_argument('--gpu', default=0, type=int, help='GPU ID to use (None for CPU)')
    parser.add_argument('--exp', default=None, type=str, help='your experiment name')


    # args parse
    args = parser.parse_args()
    feature_dim, temperature, k = args.feature_dim, args.temperature, args.k
    batch_size, epochs = args.batch_size, args.epochs
    gpu_id = args.gpu
    exp = args.exp

    # Setting the device
    device = torch.device(f"cuda:{gpu_id}" if gpu_id is not None and torch.cuda.is_available() else "cpu")
    print(f"Using device: {device}")

    # data prepare
    train_data = utils.CIFAR10Pair(root='data', train=True, transform=utils.train_transform, download=True)
    train_loader = DataLoader(train_data, batch_size=batch_size, shuffle=True, num_workers=16, pin_memory=True,
                            drop_last=True)
    memory_data = utils.CIFAR10Pair(root='data', train=True, transform=utils.test_transform, download=True)
    memory_loader = DataLoader(memory_data, batch_size=batch_size, shuffle=False, num_workers=16, pin_memory=True)
    test_data = utils.CIFAR10Pair(root='data', train=False, transform=utils.test_transform, download=True)
    test_loader = DataLoader(test_data, batch_size=batch_size, shuffle=False, num_workers=16, pin_memory=True)

    # model setup and optimizer config
    model = ProbabilisticModel(feature_dim).to(device)
    flops, params = profile(model, inputs=(torch.randn(1, 3, 32, 32).to(device),))
    flops, params = clever_format([flops, params])
    print('# Model Params: {} FLOPs: {}'.format(params, flops))
    optimizer = optim.Adam(model.parameters(), lr=1e-3, weight_decay=1e-6)
    c = len(memory_data.classes)

    # training loop
    results = {'train_loss': [], 'test_acc@1': [], 'test_acc@5': []}
    save_name_pre_ = '{}_{}_{}_{}_{}_{}'.format(feature_dim, temperature, k, batch_size, epochs, exp)
    if not os.path.exists('results'):
        os.mkdir('results')
    best_acc = 0.0
    for epoch in range(1, epochs + 1):
        save_name_pre = '{}_{}_{}_{}_{}_{}'.format(feature_dim, temperature, k, batch_size, epoch, exp)
        train_loss = train(model, train_loader, optimizer)
        results['train_loss'].append(train_loss)
        test_acc_1, test_acc_5 = test(model, memory_loader, test_loader)
        results['test_acc@1'].append(test_acc_1)
        results['test_acc@5'].append(test_acc_5)
        # save statistics
        data_frame = pd.DataFrame(data=results, index=range(1, epoch + 1))
        data_frame.to_csv('results/{}_statistics.csv'.format(save_name_pre_), index_label='epoch')
        if epoch % 50 == 0:
        # if test_acc_1 > best_acc:
        #     best_acc = test_acc_1
        ### save every 50 epochs
            torch.save(model.state_dict(), 'results/{}_model.pth'.format(save_name_pre))
