import os
from collections import OrderedDict, namedtuple
from typing import List

import torch
import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F
from torch.autograd import Variable

import sys
import numpy as np

__all__ = ["ResNet", "resnet18", "resnet34", "resnet50", "lipschitz_cnn", "small_cnn"]

from torch.nn.utils import spectral_norm

from torchvision import models


def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(
        in_planes,
        out_planes,
        kernel_size=3,
        stride=stride,
        padding=dilation,
        groups=groups,
        bias=False,
        dilation=dilation,
    )


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(
            self,
            inplanes,
            planes,
            stride=1,
            downsample=None,
            groups=1,
            base_width=64,
            dilation=1,
            norm_layer=None,
    ):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError("BasicBlock only supports groups=1 and base_width=64")
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):
    def __init__(
            self,
            block,
            layers,
            num_classes=10,
            zero_init_residual=False,
            groups=1,
            width_per_group=64,
            replace_stride_with_dilation=None,
            norm_layer=None,
    ):
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError(
                "replace_stride_with_dilation should be None "
                "or a 3-element tuple, got {}".format(replace_stride_with_dilation)
            )
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(
            3, self.inplanes, kernel_size=3, stride=1, padding=1, bias=False
        )

        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(
            block, 128, layers[1], stride=2, dilate=replace_stride_with_dilation[0]
        )
        self.layer3 = self._make_layer(
            block, 256, layers[2], stride=2, dilate=replace_stride_with_dilation[1]
        )
        self.layer4 = self._make_layer(
            block, 512, layers[3], stride=2, dilate=replace_stride_with_dilation[2]
        )
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode="fan_out", nonlinearity="relu")
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, stride=1, dilate=False):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(
            block(
                self.inplanes,
                planes,
                stride,
                downsample,
                self.groups,
                self.base_width,
                previous_dilation,
                norm_layer,
            )
        )
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(
                block(
                    self.inplanes,
                    planes,
                    groups=self.groups,
                    base_width=self.base_width,
                    dilation=self.dilation,
                    norm_layer=norm_layer,
                )
            )

        return nn.Sequential(*layers)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.reshape(x.size(0), -1)
        x = self.fc(x)

        return x


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(
            self,
            inplanes,
            planes,
            stride=1,
            downsample=None,
            groups=1,
            base_width=64,
            dilation=1,
            norm_layer=None,
    ):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.0)) * groups
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class LipschitzSpectralNorm(nn.Module):
    def __init__(self, module, lipschitz_constant=1.0):
        super(LipschitzSpectralNorm, self).__init__()
        self.module = spectral_norm(module)
        self.lipschitz_constant = lipschitz_constant

    def apply_lipschitz_constraint(self):
        weight = self.module.weight
        u = self.module.weight_u
        v = self.module.weight_v
        sigma = torch.dot(u, torch.mv(weight.view(weight.size(0), -1), v))
        factor = sigma / self.lipschitz_constant
        if factor > 1.0:
            with torch.no_grad():
                self.module.weight.copy_(weight / factor)

    def forward(self, x):
        return self.module(x)


class SmallCNN(nn.Module):
    def __init__(self, num_classes=10):
        super(SmallCNN, self).__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1)
        self.conv2 = nn.Conv2d(64, 128, kernel_size=3, stride=1, padding=1)
        self.conv3 = nn.Conv2d(128, 256, kernel_size=3, stride=1, padding=1)
        self.fc1 = nn.Linear(256 * 4 * 4, 512)
        self.fc2 = nn.Linear(512, num_classes)

        self.pool = nn.MaxPool2d(2, 2)

    def forward(self, x):
        x = self.pool(F.relu(self.conv1(x)))
        x = self.pool(F.relu(self.conv2(x)))
        x = self.pool(F.relu(self.conv3(x)))
        x = x.view(-1, 256 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.fc2(x)
        return x


def small_cnn(num_classes=10, lipschitz_constant=1.0):
    return SmallCNN(num_classes=num_classes)


class LipschitzCNN(nn.Module):
    def __init__(self, num_classes=10, lipschitz_constant=1.0):
        super(LipschitzCNN, self).__init__()
        self.lipschitz_constant = lipschitz_constant
        self.conv1 = LipschitzSpectralNorm(nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1),
                                           lipschitz_constant=self.lipschitz_constant)
        self.conv2 = LipschitzSpectralNorm(nn.Conv2d(64, 128, kernel_size=3, stride=1, padding=1),
                                           lipschitz_constant=self.lipschitz_constant)
        self.conv3 = LipschitzSpectralNorm(nn.Conv2d(128, 256, kernel_size=3, stride=1, padding=1),
                                           lipschitz_constant=self.lipschitz_constant)
        self.fc1 = LipschitzSpectralNorm(nn.Linear(256 * 4 * 4, 512), lipschitz_constant=self.lipschitz_constant)
        self.fc2 = LipschitzSpectralNorm(nn.Linear(512, num_classes), lipschitz_constant=self.lipschitz_constant)

        self.pool = nn.MaxPool2d(2, 2)

    def forward(self, x):
        x = self.pool(F.relu(self.conv1(x)))
        x = self.pool(F.relu(self.conv2(x)))
        x = self.pool(F.relu(self.conv3(x)))
        x = x.view(-1, 256 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.fc2(x)
        return x

    def apply_lipschitz_constraints(self):
        self.conv1.apply_lipschitz_constraint()
        self.conv2.apply_lipschitz_constraint()
        self.conv3.apply_lipschitz_constraint()
        self.fc1.apply_lipschitz_constraint()
        self.fc2.apply_lipschitz_constraint()


def lipschitz_cnn(num_classes=10, lipschitz_constant=1.0):
    return LipschitzCNN(num_classes=num_classes, lipschitz_constant=lipschitz_constant)


def _resnet(arch: str, block, layers: List[int], pretrained: bool, progress: bool, device: str, **kwargs):
    model = ResNet(block, layers, **kwargs)
    if pretrained:
        script_dir = os.path.dirname(__file__)
        state_dict = torch.load(
            script_dir + "/state_dicts/" + arch + ".pt", map_location=device
        )
        model.load_state_dict(state_dict)
    return model


# def resnet18(pretrained=False, progress=True, device="cpu", **kwargs):
#     """Constructs a ResNet-18 model.
#     Args:
#         pretrained (bool): If True, returns a model pre-trained on ImageNet
#         progress (bool): If True, displays a progress bar of the download to stderr
#     """
#     return _resnet(
#         "resnet18", BasicBlock, [2, 2, 2, 2], pretrained, progress, device, **kwargs
#     )


# def resnet34(pretrained=False, progress=True, device="cpu", **kwargs):
#     """Constructs a ResNet-34 model.
#     Args:
#         pretrained (bool): If True, returns a model pre-trained on ImageNet
#         progress (bool): If True, displays a progress bar of the download to stderr
#     """
#     return _resnet(
#         "resnet34", BasicBlock, [3, 4, 6, 3], pretrained, progress, device, **kwargs
#     )


def resnet50(pretrained: bool = False, progress: bool = True, device: str = "cpu", **kwargs):
    """Constructs a ResNet-50 model.
    Args:
        :param pretrained: If True, returns a model pre-trained on ImageNet
        :param progress: If True, displays a progress bar of the download to stderr
        :param device:
    """
    return _resnet(
        "resnet50", Bottleneck, [3, 4, 6, 3], pretrained, progress, device, **kwargs
    )


def resnet101(pretrained: bool = False, progress: bool = True, device: str = "cpu", **kwargs):
    """Constructs a ResNet-50 model.
    Args:
        :param pretrained: If True, returns a model pre-trained on ImageNet
        :param progress: If True, displays a progress bar of the download to stderr
        :param device:
    """
    return _resnet(
        "resnet101", Bottleneck, [3, 4, 23, 3], pretrained, progress, device, **kwargs
    )


def modify_resnet34_first_layer(model, num_input_channels):
    """Modify the first convolutional layer of ResNet-34 to accommodate different input channels."""
    if num_input_channels != 3:
        model.conv1 = nn.Conv2d(num_input_channels, 64, kernel_size=7, stride=2, padding=3, bias=False)
    return model


def resnet34(pretrained: bool = False, progress: bool = True, device: str = "cpu",
             num_input_channels: int = 3, **kwargs):
    """Constructs a ResNet-34 model with a customizable number of input channels.

    Args:
        :param num_input_channels: Number of input channels for the model (default is 3 for RGB images)
        :param pretrained: If True, returns a model pre-trained on ImageNet
        :param progress: If True, displays a progress bar of the download to stderr
        :param device:
    """
    model = models.resnet34(pretrained=pretrained, progress=progress, **kwargs)
    model = modify_resnet34_first_layer(model, num_input_channels)
    return model.to(device)


def resnet18(pretrained: bool = False, progress: bool = True, device: str = "cpu",
             num_input_channels: int = 3, **kwargs):
    """Constructs a ResNet-18 model with a customizable number of input channels.

    Args:
        :param num_input_channels: Number of input channels for the model (default is 3 for RGB images)
        :param pretrained: If True, returns a model pre-trained on ImageNet
        :param progress: If True, displays a progress bar of the download to stderr
        :param device:
    """
    model = models.resnet18(pretrained=pretrained, progress=progress, **kwargs)
    model = modify_resnet34_first_layer(model, num_input_channels)
    return model.to(device)
