import functools
import hashlib
import json
import logging
import sys
import warnings
from contextlib import redirect_stdout
from datetime import datetime
from io import StringIO
from pathlib import Path
from pydoc import locate

root_dir = Path(__file__).parent.parent.resolve()  # directory of source root
root_dir_len = len(root_dir.parts)

DATA_DIR = root_dir / 'data'
MODELS_DIR = root_dir / 'models'


class ComplexEncoder(json.JSONEncoder):
    def default(self, obj):
        # Convert complex objects to strings or another suitable format
        try:
            return str(obj)
        except Exception:
            return super().default(obj)


class MultiWriter:
    def __init__(self, *writers):
        self.writers = writers

    def write(self, message):
        for writer in self.writers:
            writer.write(message)

    def flush(self):
        for writer in self.writers:
            writer.flush()


def log_prints(logfile=root_dir / "log_experiments_results.txt"):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            log_capture_string = StringIO()
            original_stdout = sys.stdout
            multi_writer = MultiWriter(original_stdout, log_capture_string)

            now = datetime.now()
            formatted_now = now.strftime("%Y-%m-%d %H:%M:%S")
            log_call = {
                "data_time": formatted_now,
                "function": func.__name__,
                "args": args,
                "kwargs": kwargs,
            }
            log_call = json.dumps(log_call, indent=2, cls=ComplexEncoder)
            with open(logfile, 'a') as f:
                f.write(log_call)

            with redirect_stdout(multi_writer):
                result = func(*args, **kwargs)
            output = log_capture_string.getvalue()
            log_output = {
                "data_time": formatted_now,
                "function": func.__name__,
                "output": output.splitlines(),
            }
            log_output = json.dumps(log_output, indent=2, cls=ComplexEncoder)
            with open(logfile, 'a') as f:
                f.write(log_output)

            return result

        return wrapper

    return decorator
