from dataclasses import asdict, dataclass
from typing import Any, DefaultDict, Dict, List, Optional, Tuple


@dataclass
class MOSDTTrainConfig:
    # self-designed args
    seq_len: int = 10

    # wandb params
    project: str = "MOSDB-baselines"
    group: str = None
    name: Optional[str] = None
    prefix: Optional[str] = "MOSDT"
    suffix: Optional[str] = ""
    log_root_dir: Optional[str] = "../logs"
    verbose: bool = True

    # dataset params
    outliers_percent: float = None
    noise_scale: float = None
    inpaint_ranges: Tuple[Tuple[float, float], ...] = None
    epsilon: float = None
    density: float = 1
    same_r: bool = False
    same_c: bool = False

    # model params
    embedding_dim: int = 128
    num_layers: int = 2
    total_layers: int = 3
    num_heads: int = 8
    num_teacher_heads: int = 8
    action_head_layers: int = 1
    attention_dropout: float = 0.1
    residual_dropout: float = 0.1
    embedding_dropout: float = 0.1
    time_emb: bool = True
    fix_att: bool = False
    decision_heads: int = 1

    # training params
    task: str = "FreightFrankaCloseDrawer"
    dataset: str = None
    learning_rate: float = 1e-4
    betas: Tuple[float, float] = (0.9, 0.999)
    weight_decay: float = 1e-4
    clip_grad: Optional[float] = 0.25
    lr_warmup_steps: int = 500
    reward_scale: float = 0.1
    cost_scale: float = 1
    batch_size: int = 64
    num_workers: int = 8
    update_steps: int = 100000
    train_epoch: int = 1
    ct_max: float = 70
    sd_weight: float = 0.5
    feat_sd_weight: float = 0.5
    cost_classify: bool = True
    be: bool = True
    sd: bool = True
    ps: bool = True

    # evaluation params
    eval_episodes: int = 1
    eval_every: int = 2500
    cost_limit: int = 25
    save_model: bool = False

    # general params
    seed: int = 0
    device: str = "cuda:0"
    threads: int = 6

    # augmentation param
    pf_sample: bool = False
    beta: float = 1.0
    deg: int = 1
    augment_percent: float = 0.2
    max_npb: int = 10
    min_npb: int = 2

    # model mode params
    cost_transform: bool = False
    add_cost_feat: bool = False
    mul_cost_feat: bool = False
    cat_cost_feat: bool = False
    cost_reverse: bool = False

    # pf only mode param
    pf_only: bool = False
    rmin: float = 300
    cost_bins: int = 3
    npb: int = 5
    cost_sample: bool = False
    linear: bool = True  # linear or inverse
    start_sampling: bool = False
    prob: float = 0.2
    stochastic: bool = True
    init_temperature: float = 0.1
    no_entropy: bool = False

    # random augmentation
    random_aug: float = 0
    aug_rmin: float = 400
    aug_rmax: float = 500
    aug_cmin: float = -2
    aug_cmax: float = 25
    cgap: float = 5
    rstd: float = 1
    cstd: float = 0.2


@dataclass
class MOSDTSafety2x4AntVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety4x2AntVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety2x3HalfCheetahVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety6x1HalfCheetahVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety2x3Walker2dVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety3x1HopperVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety2x1SwimmerVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafety98HumanoidVelocityv0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafetyAntMultiGoal1v0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafetyPointMultiGoal1v0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafetyAntMultiGoal2v0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTSafetyPointMultiGoal2v0Config(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTShadowHandOverSafejointConfig(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTShadowHandOverSafefingerConfig(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTShadowHandCatchOver2UnderarmSafejointConfig(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTShadowHandCatchOver2UnderarmSafefingerConfig(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTFreightFrankaCloseDrawerConfig(MOSDTTrainConfig):
    pass

@dataclass
class MOSDTFreightFrankaPickAndPlaceConfig(MOSDTTrainConfig):
    pass


MOSDT_DEFAULT_CONFIG = {
    # bullet_safety_gym
    "SafetyAntMultiGoal1-v0": MOSDTSafetyAntMultiGoal1v0Config,
    "SafetyPointMultiGoal1-v0": MOSDTSafetyPointMultiGoal1v0Config,
    "SafetyAntMultiGoal2-v0": MOSDTSafetyAntMultiGoal2v0Config,
    "SafetyPointMultiGoal2-v0": MOSDTSafetyPointMultiGoal2v0Config,
    # safety_gymnasium
    "Safety2x4AntVelocity-v0": MOSDTSafety2x4AntVelocityv0Config,
    "Safety4x2AntVelocity-v0": MOSDTSafety4x2AntVelocityv0Config,
    "Safety2x3HalfCheetahVelocity-v0": MOSDTSafety2x3HalfCheetahVelocityv0Config,
    "Safety6x1HalfCheetahVelocity-v0": MOSDTSafety6x1HalfCheetahVelocityv0Config,
    "Safety2x3Walker2dVelocity-v0": MOSDTSafety2x3Walker2dVelocityv0Config,
    "Safety3x1HopperVelocity-v0": MOSDTSafety3x1HopperVelocityv0Config,
    "Safety2x1SwimmerVelocity-v0": MOSDTSafety2x1SwimmerVelocityv0Config,
    "Safety98HumanoidVelocity-v0": MOSDTSafety98HumanoidVelocityv0Config,
    # safe_isaac_gym
    "ShadowHandOver_Safe_joint": MOSDTShadowHandOverSafejointConfig,
    "ShadowHandOver_Safe_finger": MOSDTShadowHandOverSafefingerConfig,
    "ShadowHandCatchOver2Underarm_Safe_joint": MOSDTShadowHandCatchOver2UnderarmSafejointConfig,
    "ShadowHandCatchOver2Underarm_Safe_finger": MOSDTShadowHandCatchOver2UnderarmSafefingerConfig,
    "FreightFrankaCloseDrawer": MOSDTFreightFrankaCloseDrawerConfig,
    "FreightFrankaPickAndPlace": MOSDTFreightFrankaPickAndPlaceConfig,
}
