class TCR:
    """
    Define a Meaningful TCR Object that is used to store TCR information

    Args:
        cdr3a (str, optional): The CDR3 alpha sequence. Defaults to None.
        cdr3b (str, optional): The CDR3 beta sequence. Defaults to None.
        trav (str, optional): The TRAV gene. Defaults to None.
        trbv (str, optional): The TRBV gene. Defaults to None.
        traj (str, optional): The TRAJ gene. Defaults to None.
        trbj (str, optional): The TRBJ gene. Defaults to None.
        trad (str, optional): The TRAD gene. Defaults to None.
        trbd (str, optional): The TRBD gene. Defaults to None.
        pMHC (str, optional): The pMHC recognized by this TCR. Defaults to None.
        tcra_full (str, optional): The full alpha TCR sequence. Defaults to None.
        tcrb_full (str, optional): The full beta TCR sequence. Defaults to None.
        reference (str, optional): The reference for this TCR. Defaults to None.

    Returns:
        TCR Object: A TCR object that can be used to store TCR information
    """
    def __init__(self,
                cdr3a=None,
                cdr3b=None,
                trav=None,
                trbv=None,
                traj=None,
                trbj=None,
                trad=None,
                trbd=None,
                pMHC=None,
                tcra_full=None,
                tcrb_full=None,
                reference=None
                ):
        self.cdr3a = cdr3a
        self.cdr3b = cdr3b
        self.trav = trav
        self.trbv = trbv
        self.traj = traj
        self.trbj = trbj
        self.trad = trad
        self.trbd = trbd
        self.pMHCs = set([pMHC]) if pMHC is not None else set()
        self.tcra_full = tcra_full
        self.tcrb_full = tcrb_full
        self.references = set([reference]) if reference is not None else set()

    def __repr__(self):
        return f'{self.cdr3b}_{self.trbv}_{self.trbj}_{self.cdr3a}'
        
    def __str__(self):
        return f'{self.cdr3b}_{self.trbv}_{self.trbj}'

    def __eq__(self, other):
        """
        Check if two TCR objects are functionally equivalent
        """
        return self.cdr3b == other.cdr3b and self.trbv == other.trbv and self.trbj == other.trbj and self.cdr3a == other.cdr3a
    
    def __hash__(self):
        """
        Hash the TCR object
        """
        return hash((self.cdr3b, self.trbv, self.trbj, self.cdr3a))
    
    def add_pMHC(self, pMHC):
        """
        Add a pMHC to the set of cognate pMHCs recognized by this TCR

        Args:
            pMHC (pMHC Object): The pMHC to add.
        """
        self.pMHCs.add(pMHC)

    def add_reference(self, reference):
        """
        Add a reference to the set of references for this TCR

        Args:
            reference (str): The reference to add
        """
        self.references.add(reference)

    def get_pMHCs(self):
        """
        Get the set of pMHCs recognized by this TCR

        Returns:
            set: The set of pMHCs recognized by this TCR
        """
        return self.pMHCs
    
    def get_references(self):
        """
        Get the set of references for this TCR

        Returns:
            set: The set of references for this TCR
        """
        return self.references
    
