
import torch
import torch.nn as nn
from src.models.jcgel.layers.layer import JCGConv2d
from src.models.jcgel.layers.utils import CRBatchNorm2d, EquivariantSpatialPool, GlobalAvgPoolCRHW
from src.models.jcgel.layers.jcgel_resnet_block import JCGConvBlock
class CGEResNet(nn.Module):
    """CGEConv2d 기반 완전한 ResNet-50 아키텍처"""

    def __init__(self,
                 block,
                 num_blocks,
                 num_classes,
                 config,
                 # c_rotations,
                 # g_rotations,
                 n_flip=0,
                 temperature=0.01,
                 soft=True):
        super().__init__()
        self.in_channels = 64

        # self.c_rot = c_rotations
        # self.g_rot = g_rotations
        self.c_rot = config.c_rot
        self.g_rot = config.g_rot
        self.n_flip = config.n_flip
        self.temp = config.temperature
        # self.normalize = config.normalization
        self.soft =config.soft
        # Stage 1: Lifting Layer
        self.conv1 = JCGConv2d(in_channels=3, out_channels=64, kernel_size=7, stride=2, padding=3,
                               in_c_rotations=1, out_c_rotations=self.c_rot, g_rotations=self.g_rot,
                               is_lifting=True, bias=False,
                               n_flip=self.n_flip, temperature=self.temp, soft=self.soft)
        self.bn1 = CRBatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = EquivariantSpatialPool(kernel_size=3, stride=2)

        # Stages 2-5
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)

        # Classifier
        self.avgpool = GlobalAvgPoolCRHW()
        self.fc = nn.Linear(512 * block.expansion, num_classes)

    def _make_layer(self, block, out_channels, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for s in strides:
            layers.append(block(self.in_channels, out_channels, s, self.c_rot, self.g_rot))
            self.in_channels = out_channels * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, analysis=False):

        out = self.conv1(x)
        out = self.bn1(out.clone())
        out = self.relu(out)
        out = self.maxpool(out)

        # Stages 2-5
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.avgpool(out)

        out = torch.flatten(out, 1)
        if analysis:
            return out
        out = self.fc(out)

        return out

def JCGResNet18(num_classes=1000, config=None):
    """ResNet-18 모델을 생성합니다."""
    return CGEResNet(JCGConvBlock, [2, 2, 2, 2], num_classes, config)

def JCGResNet34(num_classes=1000, config=None):
    """ResNet-34 모델을 생성합니다."""
    return CGEResNet(JCGConvBlock, [3, 4, 6, 3], num_classes, config)

def JCGResNet50(num_classes=1000, config=None):
    """CGE-ResNet-50 모델 생성을 위한 헬퍼 함수"""
    return CGEResNet(JCGConvBlock, [3, 4, 6, 3], num_classes, config)
