%% clearing
clear all
close all
clc

%% problem/simulation parameters
h = 1e-2; %discretization stepsize (verlet method)
T = 200; %final time
K = floor(T/h); %total number of iterations integrator
H =[2*1e-2,0*1e-3;0*1e-3,5*1e-3]; % f = x^T H x (2 in the derivative!)
k1 = floor(K/6); %beginning point for curve subset
k2 = floor(0.9*K/2); % ending point for curve subset
t1 = h*k1;
t2 = h*k2;

%% perturbation settings
periods = 4; %how wiggly
amplitude = 0.005; %how big

%% initialization
x = zeros(2,K);
v = zeros(2,K);
x(:,1) = [1;1];

%% simulation of NAG-ODE (= su, boyd, candes paper)
for i = 2:K
    v(:,i) = v(:,i-1) - h * 3/(h*i) * v(:,i-1) - 2*H*x(:,i-1) * h;
    x(:,i) = x(:,i-1) + h * v(:,i-1);
end


%% plotting solution to NAG-ODE (and considered curve)
XX=-0.5:0.005:0.5;
YY=-0.5:0.005:0.5;
[X,Y] = meshgrid(XX,YY);
Z = H(1,1)*X.^2 + H(2,2)*Y.^2 + H(1,2)*X.*Y+ H(2,1)*X.*Y;
contour(X,Y,Z,100,'Color',[0.7,0.7,0.7]);hold on;
xlim([-0.16,0.16])
ylim([-0.16,0.16])
pbaspect([1 1 1])
plot(x(1,:),x(2,:),'-.','Linewidth',1,'Color',[0.325 0.518 0.812]);hold on
h1=plot(x(1,k1:k2),x(2,k1:k2),'Linewidth',3,'Color',[0.325 0.518 0.812]);hold on
plot(0,0,'-kh','MarkerSize', 20,'MarkerFaceColor',[0.839 0.78 0.078]);
xlabel('$x_1$','FontSize',20,'Interpreter','Latex')
ylabel('$x_2$','FontSize',20,'Interpreter','Latex')

%% cutting to the useful part of the trajectory (between k1 and k2)
x=x(:,k1:k2);
v=v(:,k1:k2);

Kdiff = size(x,2); %k2-k1+1

%% generating perturbation (just a sin wave)
pert = -amplitude*sin(2*pi*periods*h*(0:Kdiff-1)/(h*Kdiff));
d_pert = -amplitude*((2*pi*periods)/(Kdiff*h))*cos(2*pi*periods*(0:Kdiff-1)/Kdiff);

x_pert = x+[pert;pert];
v_pert = v+[d_pert;d_pert];

h2=plot(x_pert(1,:),x_pert(2,:),'Linewidth',3,'Color',[207/255 132/255 83/255]);hold on
plot(x(1,1),x(2,1),'-ko','MarkerSize', 10,'MarkerFaceColor',[0.325 0.518 0.812]);
plot(x(1,end),x(2,end),'-ko','MarkerSize', 10,'MarkerFaceColor',[0.325 0.518 0.812]);
l=legend([h1,h2],'$x(t)$','$x(t) + \epsilon h(t)$');
l.set('FontSize',26);
l.set('Interpreter','latex');

%% Computing the action for the two curves

%computing f
f = zeros(1,Kdiff);
for i = 1:Kdiff
    f(i) = x(:,i)'*H*x(:,i);
end

%computing f (perturbed
f_pert = zeros(1,Kdiff);
for i = 1:Kdiff
    f_pert(i) = x_pert(:,i)'*H*x_pert(:,i);
end

%computing lagrangians
L = (h*(k1:k2)).^3 .* ( f - 0.5*vecnorm(v,2));
L_pert = (h*(k1:k2)).^3 .* ( f_pert - 0.5*vecnorm(v_pert,2));

%computing actions
S = h*sum(L);
S_pert = h*sum(L_pert);

%% result
disp(['Action difference:',num2str(S_pert-S)])

