import subprocess
import sys
import argparse
from pathlib import Path
from typing import Optional

FILE_PATH = Path(__file__).absolute()
BASE_DIR = FILE_PATH.parent.parent
sys.path.insert(0, str(BASE_DIR))

from src.configs.config import BASE_DIR
from src.configs.logger import get_logger
from src.models.generator import ContentGenerator
from src.models.generator.latex_generator_coai import LatexGeneratorCoAI  
from src.models.generator.interactive_outlines_generator import InteractiveOutlinesGenerator
from src.models.generator.interactive_content_generator import InteractiveContentGenerator
from src.models.LLM import ChatAgent
from src.models.post_refine import PostRefiner
from src.modules.preprocessor.data_cleaner import DataCleaner
from src.modules.preprocessor.utils import create_tmp_config

logger = get_logger("tasks.offline_run_inter_full_coai")


class InteractiveSurveyGeneratorCoAI:
    """Complete Interactive Survey Generation System with CoAI Support"""
    
    def __init__(self, title: str, key_words: str, ref_path: str, task_id: str = None):
        self.title = title
        self.key_words = key_words
        self.ref_path = ref_path
        
        self.chat_agent = ChatAgent()
        logger.info("Initialized ChatAgent for interactive survey generation")
        
        if task_id:
            # Use existing task_id
            self.task_id = task_id
            
            # Try to load existing config
            existing_config_path = Path(f"{BASE_DIR}/outputs/{task_id}/tmp_config.json")
            if existing_config_path.exists():
                import json
                with open(existing_config_path, 'r') as f:
                    self.tmp_config = json.load(f)
                self.topic = self.tmp_config["topic"]
                logger.info(f"Loaded existing task configuration for: {task_id}")
            else:
                # Create new config with specified task_id
                self.tmp_config = create_tmp_config(title, key_words)
                # Override the generated task_id with the user-specified one
                self.tmp_config["task_id"] = task_id
                self.topic = self.tmp_config["topic"]
                
                # Save the config with the custom task_id
                from src.modules.utils import update_config
                config_path = Path(f"{BASE_DIR}/outputs/{task_id}/tmp_config.json")
                update_config(self.tmp_config, config_path)
                logger.info(f"Created new task configuration with specified task_id: {task_id}")
        else:
            # Create new task configuration
            self.tmp_config = create_tmp_config(title, key_words)
            self.task_id = self.tmp_config["task_id"]
            self.topic = self.tmp_config["topic"]
        
        logger.info(f"Initialized complete interactive CoAI survey generator for task: {self.task_id}")
        logger.info(f"Topic: {self.topic}")
    
    def display_header(self):
        """Display the application header"""
        print("\\n" + "="*80)
        print(" " * 10 + "COMPLETE INTERACTIVE SURVEY GENERATION SYSTEM - CoAI VERSION")
        print("="*80)
        print(f"Survey Title: {self.title}")
        print(f"Keywords: {self.key_words}")
        print(f"Reference Path: {self.ref_path}")
        print(f"Task ID: {self.task_id}")
        print("Version: CoAI Full (No watermarks, Interactive content, Complete features)")
        print("="*80)
    
    def get_user_confirmation(self, prompt: str, default: bool = True) -> bool:
        """Get user confirmation with default option"""
        default_text = "Y/n" if default else "y/N"
        while True:
            try:
                response = input(f"{prompt} ({default_text}): ").lower().strip()
                if not response:
                    return default
                elif response in ['y', 'yes']:
                    return True
                elif response in ['n', 'no']:
                    return False
                else:
                    print("Please enter 'y' for yes, 'n' for no, or press Enter for default.")
            except KeyboardInterrupt:
                print("\\nOperation cancelled by user.")
                return False
    
    def get_user_choice(self, prompt: str, options: list) -> str:
        """Get user choice from a list of options"""
        print(f"\\n{prompt}")
        for i, option in enumerate(options, 1):
            print(f"{i}. {option}")
        
        while True:
            try:
                choice = int(input("Choose an option: ")) - 1
                if 0 <= choice < len(options):
                    return options[choice]
                else:
                    print(f"Please enter a number between 1 and {len(options)}")
            except ValueError:
                print("Please enter a valid number")
    
    def check_latexmk_installed(self) -> bool:
        """Check if latexmk is installed"""
        try:
            subprocess.run(['latexmk', '--version'], 
                         check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            logger.debug("latexmk is installed.")
            return True
        except (subprocess.CalledProcessError, FileNotFoundError):
            logger.debug("latexmk is not installed.")
            return False
    
    def step_preprocess_references(self) -> bool:
        """Step 1: Preprocess references"""
        print("\\n" + "="*60)
        print("STEP 1: PREPROCESSING REFERENCES")
        print("="*60)
        
        if not self.get_user_confirmation("Start preprocessing references?"):
            return False
        
        try:
            print("Processing reference documents...")
            dc = DataCleaner()
            
            # Auto-detect input format and use appropriate processing method
            ref_path_obj = Path(self.ref_path)
            json_files = list(ref_path_obj.glob("*.json"))
            md_files = list(ref_path_obj.glob("*.md"))
            
            if json_files:
                print(f"Found {len(json_files)} JSON files. Using JSON input mode with enhanced bib processing...")
                dc.load_json_files_direct(self.ref_path)
                dc.run_enhanced_bib_processing(task_id=self.task_id, chat_agent=self.chat_agent)
            elif md_files:
                print(f"Found {len(md_files)} MD files. Using traditional MD input mode...")
                dc.offline_proc(task_id=self.task_id, ref_path=self.ref_path)
            else:
                print(f" No valid input files (.json or .md) found in {self.ref_path}")
                return False
                
            print("✓ References preprocessed successfully!")
            return True
        except Exception as e:
            logger.error(f"Error preprocessing references: {e}")
            print(f"✗ Error preprocessing references: {e}")
            return False
    
    def step_interactive_outline_generation(self) -> bool:
        """Step 2: Interactive outline generation"""
        print("\\n" + "="*60)
        print("STEP 2: INTERACTIVE OUTLINE GENERATION")
        print("="*60)
        
        if not self.get_user_confirmation("Start interactive outline generation?"):
            return False
        
        try:
            # Initialize the interactive outline generator
            outline_generator = InteractiveOutlinesGenerator(self.task_id)
            
            # Run the interactive outline generation
            outline = outline_generator.run_interactive()
            
            if outline:
                print("✓ Outline generation completed!")
                return True
            else:
                print("✗ Outline generation was cancelled or failed.")
                return False
                
        except Exception as e:
            logger.error(f"Error in outline generation: {e}")
            print(f"✗ Error in outline generation: {e}")
            return False
    
    def step_content_generation_choice(self) -> bool:
        """Step 3: Content generation with choice between interactive and traditional"""
        print("\\n" + "="*60)
        print("STEP 3: CONTENT GENERATION")
        print("="*60)
        
        print("Choose your content generation approach:")
        print("1. Interactive Content Generation - Full dialogue with advisor LLM")
        print("2. Traditional Content Generation - Automated generation")
        print("3. Skip this step")
        
        choice = self.get_user_choice("Which content generation approach?", 
                                    ["Interactive", "Traditional", "Skip"])
        
        if choice == "Skip":
            print("Skipped content generation")
            return False
        
        try:
            if choice == "Interactive":
                print("Starting interactive content generation with advisor LLM...")
                content_generator = InteractiveContentGenerator(task_id=self.task_id)
                content_generator.run_interactive()
                print("✓ Interactive content generation completed!")
                # Ensure post-refinement (includes bib name mapping) is executed
                print("Starting post-refinement to normalize citations and clean content...")
                post_refiner = PostRefiner(task_id=self.task_id, chat_agent=self.chat_agent)
                post_refiner.run()
                print("✓ Post-refinement completed!")
            else:  # Traditional
                print("Starting traditional content generation...")
                print("This process may take a significant amount of time.")
                content_generator = ContentGenerator(task_id=self.task_id)
                content_generator.run()
                print("✓ Traditional content generation completed!")
            
            return True
        except Exception as e:
            logger.error(f"Error in content generation: {e}")
            print(f"✗ Error in content generation: {e}")
            return False
    
    def step_post_refine(self) -> bool:
        """Step 4: Post-refinement"""
        print("\\n" + "="*60)
        print("STEP 4: POST-REFINEMENT")
        print("="*60)
        
        if not self.get_user_confirmation("Start post-refinement process?"):
            return False
        
        try:
            print("Running post-refinement...")
            post_refiner = PostRefiner(task_id=self.task_id, chat_agent=self.chat_agent)
            post_refiner.run()
            print("✓ Post-refinement completed!")
            return True
        except Exception as e:
            logger.error(f"Error in post-refinement: {e}")
            print(f"✗ Error in post-refinement: {e}")
            return False
    
    def step_from_mainbody_raw(self) -> bool:
        """Step 0: Start from existing mainbody.raw.tex file"""
        print("\\n" + "="*60)
        print("STEP 0: START FROM EXISTING MAINBODY.RAW.TEX")
        print("="*60)
        
        # Check if mainbody.raw.tex exists
        mainbody_raw_path = Path(f"{BASE_DIR}/outputs/{self.task_id}/tmp/mainbody.raw.tex")
        if not mainbody_raw_path.exists():
            print(f"✗ mainbody.raw.tex not found at: {mainbody_raw_path}")
            print("This step requires an existing mainbody.raw.tex file to continue from.")
            return False
        
        print(f"✓ Found mainbody.raw.tex at: {mainbody_raw_path}")
        print("This will skip the outline and content generation steps.")
        
        if not self.get_user_confirmation("Continue from existing mainbody.raw.tex?"):
            return False
        
        print("✓ Starting from existing mainbody.raw.tex file!")
        return True

    def step_process_mainbody(self) -> bool:
        """Step: Process mainbody.raw.tex to generate mainbody.tex"""
        print("\\n" + "="*60)
        print("STEP: PROCESS MAINBODY.RAW.TEX TO MAINBODY.TEX")
        print("="*60)
        
        # Check if mainbody.raw.tex exists
        mainbody_raw_path = Path(f"{BASE_DIR}/outputs/{self.task_id}/tmp/mainbody.raw.tex")
        if not mainbody_raw_path.exists():
            print(f"✗ mainbody.raw.tex not found at: {mainbody_raw_path}")
            return False
        
        if not self.get_user_confirmation("Process mainbody.raw.tex to generate mainbody.tex?"):
            return False
        
        try:
            print("Processing mainbody.raw.tex...")
            
            # Use the post_revise method from ContentGenerator to process the raw file
            from src.models.generator import ContentGenerator
            content_generator = ContentGenerator(task_id=self.task_id)
            
            mainbody_save_path = Path(f"{BASE_DIR}/outputs/{self.task_id}/tmp/mainbody.tex")
            papers_dir = Path(f"{BASE_DIR}/outputs/{self.task_id}/papers")
            
            content_generator.post_revise(mainbody_raw_path, mainbody_save_path, papers_dir)
            
            print("✓ mainbody.tex generated successfully!")
            print(f"✓ File saved at: {mainbody_save_path}")
            return True
            
        except Exception as e:
            logger.error(f"Error processing mainbody.raw.tex: {e}")
            print(f"✗ Error processing mainbody.raw.tex: {e}")
            return False

    def step_latex_generation(self) -> bool:
        """Step 5: CoAI LaTeX generation and compilation"""
        print("\\n" + "="*60)
        print("STEP 5: COAI LATEX GENERATION AND COMPILATION (NO WATERMARKS)")
        print("="*60)
        
        if not self.get_user_confirmation("Start CoAI LaTeX generation?"):
            return False
        
        try:
            latex_dir = Path(f"{BASE_DIR}/outputs/{self.task_id}/latex")
            refs_bib_path = latex_dir / "references.bib"
            if not refs_bib_path.exists():
                print(f"⚠ Warning: references.bib not found at: {refs_bib_path}")
                print("   Unresolved citations are likely. Ensure data preprocessing generated a BibTeX file.")
                if not self.get_user_confirmation("Continue LaTeX generation without references.bib?"):
                    return False

            print("Generating CoAI LaTeX files...")
            latex_generator = LatexGeneratorCoAI(task_id=self.task_id)  
            latex_generator.generate_full_survey()
            print("✓ CoAI LaTeX generation completed!")
            
            # Check if latexmk is available for compilation
            if self.check_latexmk_installed():
                if self.get_user_confirmation("Compile CoAI LaTeX to PDF?"):
                    print("Compiling CoAI LaTeX to PDF...")
                    latex_generator.compile_single_survey()
                    print("CoAI PDF compilation completed!")
            else:
                print("latexmk is not installed. Cannot compile to PDF.")
                print("You can manually compile the LaTeX files later.")
            
            return True
            
        except Exception as e:
            logger.error(f"Error in CoAI LaTeX generation: {e}")
            print(f"Error in CoAI LaTeX generation: {e}")
            return False


    def display_results(self):
        """Display final results and file locations"""
        output_dir = Path(f"{BASE_DIR}/outputs/{self.task_id}")
        
        print("\\n" + "="*80)
        print("COMPLETE COAI SURVEY GENERATION COMPLETED!")
        print("="*80)
        print(f"Output directory: {output_dir}")
        
        # List generated files with detailed descriptions
        key_files = [
            ("outlines.json", "Survey outline structure"),
            ("tmp/mainbody.raw.tex", "Raw survey content"),
            ("tmp/mainbody.tex", "Processed survey content"),
            ("tmp/abstract.tex", "Generated abstract"),
            ("latex/survey_coai.tex", "CoAI LaTeX source (no watermarks)"),
            ("survey_coai.pdf", "CoAI PDF output (no watermarks)"),
            ("interaction_summary.json", "Interaction summary"),
            ("conversation_history_*.json", "Conversation records"),
            ("conversation_history_*.markdown", "Readable conversation logs"),
        ]
        
        print(f"\\n\ud83d\udcc4 Generated files:")
        for filename, description in key_files:
            file_path = output_dir / filename
            if "*" in filename:
                # Handle wildcard patterns
                matching_files = list(output_dir.glob(filename))
                if matching_files:
                    for match in matching_files:
                        if match.exists():
                            print(f"   \u2705 {match.name}: {description}")
            else:
                if file_path.exists():
                    print(f"   \u2705 {filename}: {description}")
                else:
                    print(f"   \u274c {filename}: {description} (not found)")
    
        print(f"   - LaTeX source: {output_dir}/latex/survey_coai.tex")
        print(f"   - PDF output: {output_dir}/survey_coai.pdf")
        
        print(f"\\n\ud83d\udcac Interaction logs saved for reference:")
        print(f"   - JSON format: For data analysis")
        print(f"   - Markdown format: For easy reading")
    
    def run(self):
        """Run the complete interactive CoAI survey generation process"""
        self.display_header()
        
        # Check if mainbody.raw.tex exists for the start-from option
        mainbody_raw_path = Path(f"{BASE_DIR}/outputs/{self.task_id}/tmp/mainbody.raw.tex")
        has_mainbody_raw = mainbody_raw_path.exists()
        
        # Step-by-step execution with user control
        steps = [
            ("Preprocess References", self.step_preprocess_references),
            ("Interactive Outline Generation", self.step_interactive_outline_generation),
            ("Content Generation (Interactive/Traditional)", self.step_content_generation_choice),
            ("Post-refinement", self.step_post_refine),
            ("CoAI LaTeX Generation & Compilation", self.step_latex_generation)
        ]
        
        # Steps for continuing from mainbody.raw.tex (skip content generation)
        from_mainbody_steps = [
            ("Start from mainbody.raw.tex", self.step_from_mainbody_raw),
            ("Process mainbody.raw.tex to mainbody.tex", self.step_process_mainbody),
            ("Post-refinement", self.step_post_refine),
            ("CoAI LaTeX Generation & Compilation", self.step_latex_generation)
        ]
        
        print(f"\\nThe complete CoAI survey generation process consists of {len(steps)} main steps:")
        for i, (name, _) in enumerate(steps, 1):
            print(f"{i}. {name}")
        
        print(f"\\nExecution options:")
        print("1. Run all steps sequentially")
        print("2. Run steps individually (full control)")
        if has_mainbody_raw:
            print("3. Start from existing mainbody.raw.tex (skip content generation)")
            print("4. Exit")
        else:
            print("3. Exit")
        
        if has_mainbody_raw:
            choice = self.get_user_choice("What would you like to do?", 
                                        ["Run all steps", "Run individually", "Start from mainbody.raw.tex", "Exit"])
        else:
            choice = self.get_user_choice("What would you like to do?", 
                                        ["Run all steps", "Run individually", "Exit"])
        
        if choice == "Exit":
            print("\ud83d\udc4b Goodbye!")
            return
        
        completed_steps = []
        
        if choice == "Start from mainbody.raw.tex":
            # Use the simplified workflow starting from mainbody.raw.tex
            working_steps = from_mainbody_steps
            for step_name, step_func in working_steps:
                print(f"\\n{'='*20} {step_name} {'='*20}")
                if step_func():
                    completed_steps.append(step_name)
                else:
                    print(f"Step '{step_name}' failed or was skipped.")
                    if not self.get_user_confirmation("Continue with remaining steps?"):
                        break
        elif choice == "Run all steps":
            # Run all steps sequentially
            for step_name, step_func in steps:
                print(f"\\n{'='*20} {step_name} {'='*20}")
                if step_func():
                    completed_steps.append(step_name)
                else:
                    print(f"Step '{step_name}' failed or was skipped.")
                    if not self.get_user_confirmation("Continue with remaining steps?"):
                        break
        
        else:  # Run individually
            # Allow user to select which steps to run
            while True:
                available_steps = [(name, func) for name, func in steps]
                if has_mainbody_raw:
                    available_steps = [("Start from mainbody.raw.tex", self.step_from_mainbody_raw), ("Process mainbody.raw.tex to mainbody.tex", self.step_process_mainbody)] + available_steps
                options = [name for name, _ in available_steps] + ["Finish"]
                
                choice = self.get_user_choice("Which step would you like to run?", options)
                
                if choice == "Finish":
                    break
                
                # Find and run the selected step
                for step_name, step_func in available_steps:
                    if step_name == choice:
                        if step_func():
                            completed_steps.append(step_name)
                        break
        
        # Display final results
        print(f"\\nCompleted steps: {', '.join(completed_steps) if completed_steps else 'None'}")
        if completed_steps:
            self.display_results()


def parse_arguments():
    """Parse command line arguments with complete feature set"""
    parser = argparse.ArgumentParser(
        description="Complete Interactive CoAI Survey Generation System",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Create new CoAI survey from JSON files (enhanced bib processing)
  python tasks/offline_run_inter_full_coai.py --title "Machine Learning Survey" --key_words "machine learning, deep learning" --ref_path "./papers_json"
  
  # Create new CoAI survey from MD files (traditional processing)
  python tasks/offline_run_inter_full_coai.py --title "Machine Learning Survey" --key_words "machine learning, deep learning" --ref_path "./references_md"
  
  # Use existing task_id to continue or refine existing work
  python tasks/offline_run_inter_full_coai.py --title "Machine Learning Survey" --key_words "machine learning, deep learning" --ref_path "./papers_json" --task_id "existing_task_id"
  
  # Create new survey with custom task_id
  python tasks/offline_run_inter_full_coai.py --title "Machine Learning Survey" --key_words "machine learning, deep learning" --ref_path "./papers_json" --task_id "custom_task_name"
        """
    )
    
    parser.add_argument(
        "--title", 
        type=str, 
        required=True,
        help="Title of the survey paper"
    )
    
    parser.add_argument(
        "--key_words", 
        type=str, 
        required=True,
        help="Keywords for the survey (comma-separated)"
    )
    
    parser.add_argument(
        "--ref_path", 
        type=str, 
        required=True,
        help="Path to the directory containing reference documents (.json or .md files)"
    )
    
    parser.add_argument(
        "--task_id",
        type=str,
        required=False,
        help="Optional task ID to use existing task or specify custom task ID"
    )
    
    return parser.parse_args()


if __name__ == "__main__":
    try:
        args = parse_arguments()
        
        # Validate reference path
        ref_path = Path(args.ref_path)
        if not ref_path.exists():
            print(f"Error: Reference path '{ref_path}' does not exist.")
            sys.exit(1)
        
        if not ref_path.is_dir():
            print(f"Error: Reference path '{ref_path}' is not a directory.")
            sys.exit(1)
        
        # Check for input files (both JSON and MD supported)
        json_files = list(ref_path.glob("*.json"))
        md_files = list(ref_path.glob("*.md"))
        
        if not json_files and not md_files:
            print(f"Warning: No .json or .md files found in '{ref_path}'.")
            print("Supported formats:")
            print("  - JSON files: Pre-processed paper metadata with enhanced bib generation")
            print("  - MD files: Markdown documents for traditional processing")
            if not input("Continue anyway? (y/n): ").lower().startswith('y'):
                sys.exit(1)
        elif json_files:
            print(f"Found {len(json_files)} JSON files for enhanced processing")
        elif md_files:
            print(f"Found {len(md_files)} MD files for traditional processing")
        
        # Initialize and run the complete interactive CoAI generator
        generator = InteractiveSurveyGeneratorCoAI(
            title=args.title,
            key_words=args.key_words,
            ref_path=args.ref_path,
            task_id=args.task_id
        )
        
        generator.run()
        
    except KeyboardInterrupt:
        print("\\n\\nProcess interrupted by user. Goodbye!")
        sys.exit(0)
    except Exception as e:
        logger.error(f"Unexpected error: {e}")
        print(f"\\nUnexpected error occurred: {e}")
        print("Check the logs for more details.")
        sys.exit(1)