"use client"; 

import { useState, useEffect, useCallback } from "react";
import ReactPaginate from 'react-paginate';
import { Helix } from 'ldrs/react';
// import Image from 'next/image';
import 'ldrs/react/Helix.css';

// page.tsx
// Interfaces match the new unified API response
interface SearchResult {
  deviceName: string;
  applicant: string;
  submissionNumber: string;
  decisionDate: string;
  pdfLink?: string | null;
  thesis?: string;
  keywords?: string;
  concepts?: string;
  similarity?: number;
}

interface KeywordApiResponse {
  results: SearchResult[];
  message: string;
  currentPage: number;
  totalPages: number;
  totalRecords: number;
}

type AiApiResponse = SearchResult[];

const ITEMS_PER_PAGE = 20;

export default function Home() {
  const [query, setQuery] = useState("");
  const [results, setResults] = useState<SearchResult[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [message, setMessage] = useState("Enter keywords to search FDA device approvals...");
  
  const [currentPage, setCurrentPage] = useState(0); 
  const [totalPages, setTotalPages] = useState(0);

  const [aiQuery, setAiQuery] = useState("");
  const [aiResults, setAiResults] = useState<SearchResult[]>([]);
  const [allAiResults, setAllAiResults] = useState<SearchResult[]>([]); // Store all AI results
  const [aiCurrentPage, setAiCurrentPage] = useState(0); // Separate page state for AI
  const [aiTotalPages, setAiTotalPages] = useState(0); // Separate total pages for AI
  const [isAiLoading, setIsAiLoading] = useState(false);
  const [aiMessage, setAiMessage] = useState("Ask AI about FDA device approvals...");

  const [activeSearchType, setActiveSearchType] = useState<'keyword' | 'ai'>('keyword');

  // Helper function to paginate AI results client-side
  const paginateAiResults = (allResults: SearchResult[], page: number) => {
    const startIndex = page * ITEMS_PER_PAGE;
    const endIndex = startIndex + ITEMS_PER_PAGE;
    return allResults.slice(startIndex, endIndex);
  };

  const fetchResults = useCallback(async (currentQuery: string, page: number = 0) => {
    if (activeSearchType !== 'keyword' && currentQuery.trim()) {
        setActiveSearchType('keyword');
    }
    setIsLoading(true);
    setMessage("");
    setAiResults([]);
    setAllAiResults([]);
    setAiMessage("Ask AI about FDA device approvals..."); 
    setAiCurrentPage(0);
    setAiTotalPages(0);

    try {
      const response = await fetch('/api/search', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          query: currentQuery.trim(),
          mode: 'keyword',
          page: page + 1,
          limit: ITEMS_PER_PAGE,
        }),
      });
      
      const data: KeywordApiResponse = await response.json();

      if (data.results) {
        setResults(data.results);
        setCurrentPage(data.currentPage - 1); 
        setTotalPages(data.totalPages);

        if (data.results.length === 0 && !currentQuery.trim() && data.totalRecords > 0 && (data.currentPage -1) === (data.totalPages -1) && data.totalPages > 0) {
          setMessage("End of results.");
        } else if (data.results.length === 0 && currentQuery.trim()){
           setMessage(data.message || "No results found for your search.");
        } else if (data.results.length === 0 && !currentQuery.trim()){
            setMessage("No records found.");
        } else {
          setMessage(data.message || "");
        }
      } else {
        setResults([]);
        setMessage(data.message || "Failed to parse results.");
        setCurrentPage(0);
        setTotalPages(0);
      }
    } catch (error) {
      console.error(`Failed to fetch search results:`, error);
      setResults([]);
      setMessage("Failed to fetch results. Please try again.");
      setCurrentPage(0);
      setTotalPages(0);
    } finally {
      setIsLoading(false);
    }
  }, [activeSearchType]);

  useEffect(() => {
    if (activeSearchType === 'keyword' && !query.trim()) {
      fetchResults("", currentPage); 
    }
  }, [currentPage, activeSearchType, query, fetchResults]);

  // useEffect(() => {
  //   const handler = setTimeout(() => {
  //     if (activeSearchType === 'keyword' && query.trim()) {
  //       setCurrentPage(0); 
  //       fetchResults(query, 0);
  //     } else if (activeSearchType === 'keyword' && !query.trim() && results.length === 0 && currentPage === 0 ) {
  //       fetchResults("", 0);
  //     }
  //   }, 500);
  //   return () => clearTimeout(handler);
  // }, [query, activeSearchType, fetchResults, results.length, currentPage]);

  const fetchAiResults = async (currentAiQuery: string) => {
    if (!currentAiQuery.trim()) {
      setAiResults([]);
      setAllAiResults([]);
      setAiMessage("Please enter a query for AI search.");
      setAiCurrentPage(0);
      setAiTotalPages(0);
      return;
    }
    setActiveSearchType('ai');
    setIsAiLoading(true);
    setAiMessage("");
    setResults([]); 
    setMessage("");   
    setCurrentPage(0); 
    setTotalPages(0);  

    try {
      const response = await fetch('/api/search', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ query: currentAiQuery.trim(), mode: 'embedding' }),
      });
      
      if (!response.ok) {
        let errorDetails = "Failed to fetch AI results.";
        try {
            const errorData = await response.json();
            errorDetails = errorData.message || errorData.details || JSON.stringify(errorData);
        } catch { errorDetails = await response.text() || response.statusText; }
        setAiResults([]);
        setAllAiResults([]);
        setAiMessage(`Error: ${errorDetails}`);
        setAiCurrentPage(0);
        setAiTotalPages(0);
        setIsAiLoading(false);
        return;
      }

      const data: AiApiResponse = await response.json();
      setAllAiResults(data); // Store all results
      
      // Calculate pagination
      const totalPages = Math.ceil(data.length / ITEMS_PER_PAGE);
      setAiTotalPages(totalPages);
      setAiCurrentPage(0); // Reset to first page
      
      // Set first page of results
      const firstPageResults = paginateAiResults(data, 0);
      setAiResults(firstPageResults);
      
      if (data.length === 0) {
        setAiMessage("No AI results found for your query.");
      } else {
        setAiMessage("");
      }
    } catch (error) {
      console.error(`Failed to fetch AI search results:`, error);
      setAiResults([]);
      setAllAiResults([]);
      setAiMessage("Failed to fetch AI results. Please try again.");
      setAiCurrentPage(0);
      setAiTotalPages(0);
    } finally {
      setIsAiLoading(false);
    }
  };
  
  const handleKeywordSearchButtonClick = () => {
    setActiveSearchType('keyword');
    setCurrentPage(0); 
    fetchResults(query, 0);
  };

  const handleAiSearchButtonClick = () => {
    fetchAiResults(aiQuery);
  };

  const handleKeywordKeyDown = (event: React.KeyboardEvent<HTMLInputElement>) => {
    if (event.key === 'Enter') handleKeywordSearchButtonClick();
  };

  const handleAiKeyDown = (event: React.KeyboardEvent<HTMLInputElement>) => {
    if (event.key === 'Enter') handleAiSearchButtonClick();
  };

  const handlePageClick = (event: { selected: number }) => {
    if (activeSearchType === 'keyword') {
      // When a page is clicked, we should fetch results for that page
      // If there's a query, use it. Otherwise, fetch the page for the full list.
      fetchResults(query, event.selected);
    } else if (activeSearchType === 'ai') {
      // For AI results, paginate client-side
      setAiCurrentPage(event.selected);
      const newPageResults = paginateAiResults(allAiResults, event.selected);
      setAiResults(newPageResults);
    }
  };

  return (
    <>
      <style jsx global>{`
        .pagination-container { display: flex; justify-content: center; align-items: center; list-style: none; padding: 0; margin-top: 1.5rem; margin-bottom: 1.5rem; }
        .pagination-container li { margin-left: 0.5rem; margin-right: 0.5rem; }
        .pagination-container li a { padding: 0.5rem 0.75rem; border: 1px solid #d1d5db; border-radius: 0.375rem; color: #374151; cursor: pointer; text-decoration: none; }
        .pagination-container li a:hover { background-color: #f3f4f6; }
        .pagination-container li.pagination-active a { background-color: #2563eb; color: white; border-color: #2563eb; }
        .pagination-container li.pagination-active a:hover { background-color: #1d4ed8; }
        .pagination-container li.pagination-disabled a { opacity: 0.5; cursor: not-allowed; }
        .pagination-container li.pagination-disabled a:hover { background-color: transparent; }
        .pagination-previous a, .pagination-next a { font-weight: 600; }
        
        /* Purple theme for AI pagination */
        .ai-pagination .pagination-container li.pagination-active a { 
          background-color: #8b5cf6; 
          color: white; 
          border-color: #8b5cf6; 
        }
        .ai-pagination .pagination-container li.pagination-active a:hover { 
          background-color: #7c3aed; 
        }
      `}</style>

      <main className="flex min-h-screen flex-col items-center p-12 bg-gray-50">


      <div className="w-full max-w-4xl">
      <h1 className="text-4xl font-bold text-center text-gray-800 mb-2">FDA AI Device Search</h1>
          <p className="text-center text-sm text-gray-600 mb-8">
            Search against the FDA&rsquo;s list of{" "}
            <a 
              href="https://www.fda.gov/medical-devices/software-medical-device-samd/artificial-intelligence-and-machine-learning-aiml-enabled-medical-devices" 
              target="_blank" 
              rel="noopener noreferrer"
              className="text-blue-600 hover:text-blue-800 hover:underline ml-1"
            >
              AI/ML-Enabled Medical Devices
            </a>.
          </p>

          {/* Keyword Search */}
          <div className="mb-6">
            <label
              htmlFor="keyword-search"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              Keyword Search
            </label>
            <div className="flex">
              <input
                id="keyword-search"
                type="text"
                value={query}
                onChange={(e) => {
                  setQuery(e.target.value);
                  if (activeSearchType !== 'keyword' && !aiQuery.trim()) {
                    setActiveSearchType('keyword');
                  }
                }}
                onKeyDown={handleKeywordKeyDown}
                placeholder="Enter keywords (e.g., device name, applicant, technology)"
                className={`
                  w-full p-4 text-lg
                  border border-gray-300 rounded-l-lg shadow-sm
                  focus:ring-2 focus:ring-blue-500 focus:border-blue-500
                  outline-none
                  text-gray-900            /* dark input text */
                  placeholder-gray-400     /* light placeholder */
                `}
              />
              <button
                onClick={handleKeywordSearchButtonClick}
                className="p-4 text-lg bg-blue-600 text-white rounded-r-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-1"
                disabled={isLoading && activeSearchType === 'keyword'}
              >
                Search
              </button>
            </div>
          </div>

          {/* AI Semantic Search */}
          <div className="mb-8">
            <label
              htmlFor="ai-search"
              className="block text-sm font-medium text-gray-700 mb-1"
            >
              AI Semantic Search
            </label>
            <div className="flex">
              <input
                id="ai-search"
                type="text"
                value={aiQuery}
                onChange={(e) => {
                  setAiQuery(e.target.value);
                  if (e.target.value.trim()) {
                    setActiveSearchType('ai');
                  } else if (!query.trim()) {
                    setActiveSearchType('keyword');
                  }
                }}
                onKeyDown={handleAiKeyDown}
                placeholder="Ask AI (e.g., 'devices for sleep apnea', 'innovations in cardiac stents')"
                className={`
                  w-full p-4 text-lg
                  border border-purple-300 rounded-l-lg shadow-sm
                  focus:ring-2 focus:ring-purple-500 focus:border-purple-500
                  outline-none
                  text-gray-900
                  placeholder-gray-400
                `}
                style={{
                  backgroundImage: 'linear-gradient(to right, #e9defa, #f3e8ff)',
                  borderColor: '#c084fc',
                }}
              />
              <button
                onClick={handleAiSearchButtonClick}
                className="p-4 text-lg bg-purple-600 text-white rounded-r-lg hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-1"
                disabled={isAiLoading && activeSearchType === 'ai'}
                style={{ backgroundImage: 'linear-gradient(to right, #9333ea, #7e22ce)' }}
              >
                Ask AI
              </button>
            </div>
          </div>
         
          {/* Unified Search Results Section */}
          <div className="mb-12">
            {activeSearchType === 'keyword' && message && <p className="text-center text-gray-600 mb-4">{message}</p>}
            {activeSearchType === 'ai' && aiMessage && <p className="text-center text-gray-600 mb-4">{aiMessage}</p>}

            {/* Pagination for both keyword and AI results */}
            {((activeSearchType === 'keyword' && totalPages > 1) || (activeSearchType === 'ai' && aiTotalPages > 1)) && (
              <div className={activeSearchType === 'ai' ? 'ai-pagination' : ''}>
                <ReactPaginate
                  previousLabel={'<'} nextLabel={'>'} breakLabel={'...'}
                  pageCount={activeSearchType === 'keyword' ? totalPages : aiTotalPages}
                  marginPagesDisplayed={2} pageRangeDisplayed={3}
                  onPageChange={handlePageClick}
                  containerClassName={'pagination-container'}
                  pageClassName={'pagination-page'} pageLinkClassName={'pagination-page-link'}
                  previousClassName={'pagination-previous'} previousLinkClassName={'pagination-previous-link'}
                  nextClassName={'pagination-next'} nextLinkClassName={'pagination-next-link'}
                  breakClassName={'pagination-break'} breakLinkClassName={'pagination-break-link'}
                  activeClassName={'pagination-active'} disabledClassName={'pagination-disabled'}
                  forcePage={activeSearchType === 'keyword' ? currentPage : aiCurrentPage}
                />
              </div>
            )}

            {((activeSearchType === 'keyword' && (results.length > 0 || isLoading)) || (activeSearchType === 'ai' && (aiResults.length > 0 || isAiLoading))) && (
              <div className="bg-white shadow-md rounded-lg overflow-x-auto mt-4">
                <table className="min-w-full divide-y divide-gray-200">
                  {activeSearchType === 'keyword' ? (
                    <thead className="bg-gray-100">
                      <tr>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/2">Device Details</th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/4">Thesis</th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/4">Key Concepts</th>
                      </tr>
                    </thead>
                  ) : ( 
                    <thead className="bg-purple-50">
                      <tr>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-purple-700 uppercase tracking-wider w-1/3">Device Details</th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-purple-700 uppercase tracking-wider w-1/3">AI Summary</th>
                        {/* <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-purple-700 uppercase tracking-wider w-1/3">Similarity Score</th> */}
                      </tr>
                    </thead>
                  )}
                  <tbody className="bg-white divide-y divide-gray-200">
                    {activeSearchType === 'keyword' && isLoading && (
                      <tr><td colSpan={3} className="px-6 py-10 text-center"><div className="flex justify-center items-center"><Helix size={70} color="#2563eb" speed={1.2} /></div></td></tr>
                    )}
                    {activeSearchType === 'ai' && isAiLoading && (
                      <tr><td colSpan={3} className="px-6 py-10 text-center"><div className="flex justify-center items-center"><Helix size={70} color="#8b5cf6" speed={1.2} /></div></td></tr>
                    )}

                    {activeSearchType === 'keyword' && !isLoading && results.map((item, index) => (
                      <tr key={`keyword-${index}`} className={index % 2 === 0 ? "bg-white" : "bg-gray-50"}>
                        <td className="px-6 py-4 text-sm text-gray-900 align-top">
                          <div className="font-medium">{item.deviceName}</div>
                          <div className="text-gray-600">Applicant: {item.applicant}</div>
                          <div className="text-gray-600">
                            Submission: {item.pdfLink && (item.pdfLink.startsWith('http://') || item.pdfLink.startsWith('https://')) ? (
                              <a href={item.pdfLink} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:text-blue-800 hover:underline">
                                {item.submissionNumber}
                              </a>
                            ) : ( item.submissionNumber )}
                          </div>
                          <div className="text-gray-600">Decision Date: {item.decisionDate}</div>
                        </td>
                        <td className="px-6 py-4 text-sm text-gray-500 align-top">{item.thesis || 'N/A'}</td>
                        <td className="px-6 py-4 text-sm text-gray-500 align-top">{item.concepts || 'N/A'}</td>
                      </tr>
                    ))}
                    
                    {activeSearchType === 'ai' && !isAiLoading && aiResults.map((item, index) => (
                      <tr key={`ai-${index}`} className={index % 2 === 0 ? "bg-white" : "bg-purple-50/50"}>
                        <td className="px-6 py-4 text-sm text-gray-900 align-top">
                          <div className="font-medium">{item.deviceName}</div>
                          <div className="text-gray-600">Applicant: {item.applicant}</div>
                          <div className="text-gray-600">
                            Submission: {item.pdfLink && (item.pdfLink.startsWith('http://') || item.pdfLink.startsWith('https://')) ? (
                              <a href={item.pdfLink} target="_blank" rel="noopener noreferrer" className="text-purple-600 hover:text-purple-800 hover:underline">
                                {item.submissionNumber}
                              </a>
                            ) : ( item.submissionNumber )}
                          </div>
                          <div className="text-gray-600">Decision Date: {item.decisionDate}</div>
                        </td>
                        <td className="px-6 py-4 text-sm text-gray-500 align-top">{item.thesis || 'N/A'}</td>
                        {/* <td className="px-6 py-4 text-sm text-gray-500 align-top">
                          <div className="text-center">
                            <div className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                              {(item.similarity || 0).toFixed(3)}
                            </div>
                          </div>
                        </td> */}
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}

            {/* Show pagination at bottom too if there are results */}
            {((activeSearchType === 'keyword' && totalPages > 1 && results.length > 0) || (activeSearchType === 'ai' && aiTotalPages > 1 && aiResults.length > 0)) && (
              <div className={activeSearchType === 'ai' ? 'ai-pagination' : ''}>
                <ReactPaginate
                  previousLabel={'<'} nextLabel={'>'} breakLabel={'...'}
                  pageCount={activeSearchType === 'keyword' ? totalPages : aiTotalPages}
                  marginPagesDisplayed={2} pageRangeDisplayed={3}
                  onPageChange={handlePageClick}
                  containerClassName={'pagination-container'}
                  pageClassName={'pagination-page'} pageLinkClassName={'pagination-page-link'}
                  previousClassName={'pagination-previous'} previousLinkClassName={'pagination-previous-link'}
                  nextClassName={'pagination-next'} nextLinkClassName={'pagination-next-link'}
                  breakClassName={'pagination-break'} breakLinkClassName={'pagination-break-link'}
                  activeClassName={'pagination-active'} disabledClassName={'pagination-disabled'}
                  forcePage={activeSearchType === 'keyword' ? currentPage : aiCurrentPage}
                />
              </div>
            )}
          </div>
        </div>

        {/* Footer with Dana-Farber Logo */}
        {/* <footer className="w-full mt-16 py-8 border-t  bg-white">
          <div className="flex flex-col items-center justify-center space-y-4">
            <div className="flex flex-col items-center space-y-3">
              <Image
                src="/DanaFarber.png"
                alt="Dana-Farber Cancer Institute"
                width={210}
                height={60}
                className="object-contain"
                priority
              />
              <div className="text-center">
                <p className="text-sm font-medium text-gray-700">
                  Powered by Dana-Farber Cancer Institute
                </p>
                <p className="text-xs text-gray-500 mt-1 mb-10">
                © 2025 Dana-Farber Cancer Institute. All rights reserved.
                </p>
              </div>
            </div>
          </div>
        </footer> */}

      </main>
    </>
  );
}
