import numpy as np
import torch
from torch.optim import Adam
import gym
import time
import pandas as pd
import spinup.algos.fuz.fuzcppo.core as core
from spinup.utils.logx import EpochLogger
from spinup.utils.mpi_pytorch import setup_pytorch_for_mpi, sync_params, mpi_avg_grads
from spinup.utils.mpi_tools import mpi_fork, mpi_avg, proc_id, mpi_statistics_scalar, num_procs


class FuzCPPOBuffer:
    """
    A buffer for storing trajectories experienced by a PPO agent interacting
    with the environment, and using Generalized Advantage Estimation (GAE-Lambda)
    for calculating the advantages of state-action pairs.
    """

    def __init__(self, obs_dim, act_dim, size, gamma=0.99, lam=0.95, outlevel=10, device="cuda:0"):
        self.obs_buf = np.zeros(core.combined_shape(size, obs_dim), dtype=np.float32)
        self.act_buf = np.zeros(core.combined_shape(size, act_dim), dtype=np.float32)
        self.adv_buf = np.zeros(size, dtype=np.float32)
        self.rew_buf = np.zeros(size, dtype=np.float32)
        self.ret_buf = np.zeros(size, dtype=np.float32)
        self.val_buf = np.zeros(size, dtype=np.float32)
        self.valupdate_buf = np.zeros(size, dtype=np.float32)
        self.logp_buf = np.zeros(size, dtype=np.float32)

        self.v_disturb_buf = np.zeros(core.combined_shape(size, outlevel+1), dtype=np.float32)
        self.fvnext_buf = np.zeros(size, dtype=np.float32)
        self.true_ret_buf = np.zeros(size, dtype=np.float32)

        self.gamma, self.lam = gamma, lam
        self.ptr, self.path_start_idx, self.max_size = 0, 0, size
        self.device = device

    def store(self, obs, act, rew, val, valupdate, logp, next_v_distrib, fuzzy_v_next):
        """
        Append one timestep of agent-environment interaction to the buffer.
        """
        assert self.ptr < self.max_size     # buffer has to have room so you can store
        self.obs_buf[self.ptr] = obs
        self.act_buf[self.ptr] = act
        self.rew_buf[self.ptr] = rew
        self.val_buf[self.ptr] = val
        self.valupdate_buf[self.ptr] = valupdate
        self.logp_buf[self.ptr] = logp
        self.v_disturb_buf[self.ptr] = next_v_distrib
        self.fvnext_buf[self.ptr] = fuzzy_v_next
        self.ptr += 1
        

    def finish_path(self, last_val=0):
        """
        Call this at the end of a trajectory, or when one gets cut off
        by an epoch ending. This looks back in the buffer to where the
        trajectory started, and uses rewards and value estimates from
        the whole trajectory to compute advantage estimates with GAE-Lambda,
        as well as compute the rewards-to-go for each state, to use as
        the targets for the value function.

        The "last_val" argument should be 0 if the trajectory ended
        because the agent reached a terminal state (died), and otherwise
        should be V(s_T), the value function estimated for the last state.
        This allows us to bootstrap the reward-to-go calculation to account
        for timesteps beyond the arbitrary episode horizon (or epoch cutoff).
        """

        path_slice = slice(self.path_start_idx, self.ptr)
        rews = np.append(self.rew_buf[path_slice], last_val)
        vals = np.append(self.val_buf[path_slice], last_val)
        
        fvals_next = self.fvnext_buf[path_slice]

        # the next two lines implement GAE-Lambda advantage calculation
        deltas = rews[:-1] + self.gamma * vals[1:] - vals[:-1]
        self.adv_buf[path_slice] = core.discount_cumsum(deltas, self.gamma * self.lam)

        self.adv_buf = self.adv_buf - self.valupdate_buf

        # the next line computes rewards-to-go, to be targets for the value function
        self.ret_buf[path_slice] = rews[:-1] + self.gamma * fvals_next
        self.true_ret_buf[path_slice] = core.discount_cumsum(rews, self.gamma)[:-1]
        
        self.path_start_idx = self.ptr

    def get(self):
        """
        Call this at the end of an epoch to get all of the data from
        the buffer, with advantages appropriately normalized (shifted to have
        mean zero and std one). Also, resets some pointers in the buffer.
        """
        assert self.ptr == self.max_size    # buffer has to be full before you can get
        self.ptr, self.path_start_idx = 0, 0
        # the next two lines implement the advantage normalization trick
        adv_mean, adv_std = mpi_statistics_scalar(self.adv_buf)
        self.adv_buf = (self.adv_buf - adv_mean) / adv_std
        data = dict(obs=self.obs_buf, act=self.act_buf, rew=self.rew_buf, ret=self.ret_buf, v_disturb=self.v_disturb_buf, t_ret=self.true_ret_buf, v=self.val_buf,
                    adv=self.adv_buf, logp=self.logp_buf)
        return {k: torch.as_tensor(v, dtype=torch.float32).to(self.device) for k,v in data.items()}


def fuzcppo(env_fn, actor_critic=core.MLPActorCritic, ac_kwargs=dict(), seed=0, device="cuda:0", level=10,
        steps_per_epoch=4000, epochs=50, gamma=0.99, clip_ratio=0.2, pi_lr=3e-4,
        vf_lr=1e-3, train_pi_iters=80, train_v_iters=80, lam=0.97, max_ep_len=100,
        target_kl=0.01, logger_kwargs=dict(), save_freq=100, fuz_freq=5,
        alpha=0.9, beta=2800.0, nu_lr=1e-3, lam_lr=1e-3, nu_start=0.0, lam_start=0.5,
        nu_delay=0.8, lam_low_bound=0.001, delay=1.0, cvar_clip_ratio=0.05
        ):
    # Special function to avoid certain slowdowns from PyTorch + MPI combo.
    setup_pytorch_for_mpi()

    # Set up logger and save configuration
    logger = EpochLogger(**logger_kwargs)
    logger.save_config(locals())

    # Random seed
    seed += 10000 * proc_id()
    torch.manual_seed(seed)
    np.random.seed(seed)

    # Instantiate environment
    env = env_fn()
    obs_dim = env.observation_space.shape
    act_dim = env.action_space.shape

    # Create actor-critic module
    ac = actor_critic(env.observation_space, env.action_space, **ac_kwargs).to(device)
    fuzzy_logic_system = core.ANFIS(out_level=level, state_dim=env.observation_space.shape[0], device=device).to(device)
    # Sync params across processes
    sync_params(ac)

    # Count variables
    var_counts = tuple(core.count_vars(module) for module in [ac.pi, ac.v])
    logger.log('\nNumber of parameters: \t pi: %d, \t v: %d\n'%var_counts)

    # Set up experience buffer
    local_steps_per_epoch = int(steps_per_epoch / num_procs())
    buf = FuzCPPOBuffer(obs_dim, act_dim, local_steps_per_epoch, gamma, lam, outlevel=level, device=device)

    # parameter of cvar
    nu = nu_start  
    cvarlam = lam_start

    # Set up function for computing PPO policy loss
    def compute_loss_pi(data):
        obs, act, adv, logp_old = data['obs'], data['act'], data['adv'], data['logp']

        # Policy loss
        pi, logp = ac.pi(obs, act)
        ratio = torch.exp(logp - logp_old)
        clip_adv = torch.clamp(ratio, 1-clip_ratio, 1+clip_ratio) * adv
        loss_pi = -(torch.min(ratio * adv, clip_adv)).mean()

        # Useful extra info
        approx_kl = (logp_old - logp).mean().item()
        ent = pi.entropy().mean().item()
        clipped = ratio.gt(1+clip_ratio) | ratio.lt(1-clip_ratio)
        clipfrac = torch.as_tensor(clipped, dtype=torch.float32).to(device).mean().item()
        pi_info = dict(kl=approx_kl, ent=ent, cf=clipfrac)

        return loss_pi, pi_info

    # Set up function for computing value loss
    def compute_loss_v(data):
        obs, ret = data['obs'], data['ret']
        return ((ac.v(obs) - ret)**2).mean()

    def compute_loss_fuzzy(data):
        obs, r, v, t_ret, v_disturb = data['obs'], data['rew'], data['v'], data['t_ret'], data['v_disturb']

        transition_noise_weight = fuzzy_logic_system(torch.as_tensor(obs, dtype=torch.float32).to(device))
        fuzzy_v_next = torch.sum(transition_noise_weight * v_disturb, dim=1, keepdim=True)
        return ((r + gamma * fuzzy_v_next - t_ret)**2).mean()

    # Set up optimizers for policy and value function
    fuzzy_lr = 3e-4
    pi_optimizer = Adam(ac.pi.parameters(), lr=pi_lr)
    vf_optimizer = Adam(ac.v.parameters(), lr=vf_lr)
    anfis_optimizer = Adam(fuzzy_logic_system.parameters(),lr=fuzzy_lr)
    eps_list = np.linspace(-0.1, 0.1, level+1)

    # Set up model saving
    logger.setup_pytorch_saver(ac)

    def update():
        data = buf.get()

        pi_l_old, pi_info_old = compute_loss_pi(data)
        pi_l_old = pi_l_old.item()
        v_l_old = compute_loss_v(data).item()

        # Train policy with multiple steps of gradient descent
        for i in range(train_pi_iters):
            pi_optimizer.zero_grad()
            loss_pi, pi_info = compute_loss_pi(data)
            kl = mpi_avg(pi_info['kl'])
            if kl > 1.5 * target_kl:
                logger.log('Early stopping at step %d due to reaching max kl.'%i)
                break
            loss_pi.backward()
            mpi_avg_grads(ac.pi)    # average grads across MPI processes
            pi_optimizer.step()

        logger.store(StopIter=i)

        # Value function learning
        for i in range(train_v_iters):
            loss_fuzzy = compute_loss_fuzzy(data)
            if epoch % fuz_freq == 0:
                anfis_optimizer.zero_grad()
                loss_fuzzy.backward()
                mpi_avg_grads(fuzzy_logic_system)
                anfis_optimizer.step()

            vf_optimizer.zero_grad()
            loss_v = compute_loss_v(data)
            loss_v.backward()
            mpi_avg_grads(ac.v)    # average grads across MPI processes
            vf_optimizer.step()

        # Log changes from update
        kl, ent, cf = pi_info['kl'], pi_info_old['ent'], pi_info['cf']
        logger.store(LossPi=pi_l_old, LossV=v_l_old, LossFuzzy=loss_fuzzy.item(),
                     KL=kl, Entropy=ent, ClipFrac=cf,
                     DeltaLossPi=(loss_pi.item() - pi_l_old),
                     DeltaLossV=(loss_v.item() - v_l_old))

    # Prepare for interaction with environment
    start_time = time.time()
    o, ep_ret, ep_len = env.reset()[0], 0, 0
    ep_risk = 0.
    # Main loop: collect experience in env and update/log each epoch
    for epoch in range(epochs):
        trajectory_num = 0
        bad_trajectory_num = 0
        # nu = nu + nu_lr * cvarlam
        cvarlam = cvarlam + lam_lr * (beta - nu)
        lam_delta = 0
        nu_delta = 0
        update_num = 0

        ep_ret = 0
        ep_true_ret = 0
        
        for t in range(local_steps_per_epoch):
            a, v, logp = ac.step(torch.as_tensor(o, dtype=torch.float32).to(device))
            next_o, r, d, infos = env.step(a)
            c = infos['constraint_violation']
            r_true = r.copy()

            repeat_time= 1
            batch_size = (level + 1) * repeat_time
            next_o_np = next_o.cpu().numpy() if isinstance(next_o, torch.Tensor) else next_o
            next_obs_batch_np = np.repeat(next_o_np.reshape(1, -1), batch_size, axis=0)

            perturbation_ranges = [(eps_list[i]-0.005, eps_list[i]+0.005) for i in range(len(eps_list))]

            noise_level_np = np.vstack([
                np.random.uniform(low=low, high=high, size=next_obs_batch_np.shape[1])
                for low, high in perturbation_ranges
                for _ in range(repeat_time)  
            ])

            noise_np = noise_level_np * next_obs_batch_np
            next_obs_batch_perb_np = next_obs_batch_np + noise_np
            next_obs_batch_perb_np = np.round(next_obs_batch_perb_np, 3)


            with torch.no_grad():
                _, next_v_np, _ = ac.step(torch.from_numpy(next_obs_batch_perb_np).float().to(device))

            next_v_distrib_np = next_v_np.reshape(level + 1, repeat_time).mean(axis=1)

            with torch.no_grad():
                transition_noise_weight_np = fuzzy_logic_system(torch.as_tensor(o, dtype=torch.float32).unsqueeze(0).to(device)).reshape(-1)

            fuzzy_v_next_np = 0
            for i in range(level + 1):
                fuzzy_v_next_np += transition_noise_weight_np[i] * next_v_distrib_np[i]

            ep_ret += r
            ep_true_ret += r_true
            ep_risk += c
            ep_len += 1
            # the num of trajectories
            trajectory_num += 1
            nu_delta += ep_ret + v - r
            updates = np.float32(0.0)
            if ep_ret + v - r < nu: 
                bad_trajectory_num += 1
                lam_delta += ep_ret + v - r
                updates = delay * cvarlam / (1 - alpha) * (nu - ep_ret - v + r)
                if updates > abs(v) * cvar_clip_ratio:
                    # print("update: ", updates)
                    updates = abs(v) * cvar_clip_ratio
                    update_num += 1
                    # print("v", v)
                updates = np.float32(updates)

            # save and log
            buf.store(o, a, r, v, updates, logp, next_v_distrib_np, fuzzy_v_next_np)
            logger.store(VVals=v)
            
            # Update obs (critical!)
            o = next_o

            timeout = ep_len == max_ep_len
            terminal = d or timeout
            epoch_ended = t==local_steps_per_epoch-1

            if terminal or epoch_ended:
                if epoch_ended and not(terminal):
                    print('Warning: trajectory cut off by epoch at %d steps.'%ep_len, flush=True)
                # if trajectory didn't reach terminal state, bootstrap value target
                if timeout or epoch_ended:
                    _, v, _ = ac.step(torch.as_tensor(o, dtype=torch.float32).to(device))
                else:
                    v = 0
                buf.finish_path(v)
                if terminal:
                    logger.store(EpRet=ep_ret, EpTrueRet=ep_true_ret, EpRisk=ep_risk, EpLen=ep_len)
                o, ep_ret, ep_len = env.reset()[0], 0, 0
                ep_risk = 0
                ep_true_ret = 0

        if bad_trajectory_num > 0:
            lam_delta = lam_delta / bad_trajectory_num
        if trajectory_num > 0:
            nu_delta = nu_delta / trajectory_num
        nu = nu_delta * nu_delay
        
        # Save model
        if (epoch % save_freq == 0) or (epoch == epochs-1):
            logger.save_state({'env': env}, epoch)

        # Perform PPO update!
        update()

        # Log info about epoch
        logger.log_tabular('Epoch', epoch)
        logger.log_tabular('EpRet', average_only=True)
        logger.log_tabular('EpTrueRet', average_only=True)
        logger.log_tabular('EpRisk', average_only=True)
        logger.log_tabular('EpLen', average_only=True)
        logger.log_tabular('VVals', with_min_and_max=True)
        logger.log_tabular('TotalEnvInteracts', (epoch+1)*steps_per_epoch)
        logger.log_tabular('LossPi', average_only=True)
        logger.log_tabular('LossV', average_only=True)
        logger.log_tabular('LossFuzzy', average_only=True)
        logger.log_tabular('DeltaLossPi', average_only=True)
        logger.log_tabular('DeltaLossV', average_only=True)
        logger.log_tabular('Entropy', average_only=True)
        logger.log_tabular('KL', average_only=True)
        logger.log_tabular('ClipFrac', average_only=True)
        logger.log_tabular('StopIter', average_only=True)
        logger.log_tabular('Time', time.time()-start_time)
        logger.dump_tabular()

        print("-" * 37)
        print("bad_trajectory_num:", bad_trajectory_num)
        print("update num:", update_num)
        print("nu:", nu)
        print("lam:", cvarlam)
        print("-" * 37, flush=True)

