# IconBank: Image Similarity Search System

A high-performance image similarity search system built with TensorRT, FAISS, and ResNet50 for efficient icon and UI element retrieval.

## Overview

IconBank is designed to perform fast similarity search on large collections of UI elements and icons. It uses ResNet50 feature extraction with TensorRT optimization and FAISS indexing for scalable similarity search across multiple datasets.

## Features

- **High Performance**: TensorRT-optimized ResNet50 feature extraction
- **Scalable Search**: FAISS-based indexing for fast similarity search
- **Multi-Dataset Support**: Handles multiple UI/icon datasets simultaneously
- **Bbox-aware Search**: Supports both full image and cropped region search
- **Batch Processing**: Efficient batch processing for large-scale operations
- **GPU Acceleration**: CUDA-optimized inference and preprocessing

## System Requirements

- Python 3.7+
- CUDA-capable GPU
- TensorRT 8.0+
- PyTorch
- OpenCV
- FAISS-GPU

## Installation

1. Install dependencies:
```bash
pip install -r requirements.txt
```

2. Install TensorRT (follow NVIDIA's official installation guide)

3. Install FAISS-GPU:
```bash
pip install faiss-gpu
```

## Project Structure

```
iconBank/
├── build_index.py                    # Build FAISS index from image datasets
├── build_index_with_crop_img.py      # Build index with bbox-aware cropping
├── search.py                         # Core search functionality
├── search_bad_imgs.py               # Search for problematic images
├── search_bad_imgs_with_icon.py     # Enhanced search with icon support
├── copy_matched_icon_data_to_new_path.py  # Data processing utilities
├── requirements.txt                 # Python dependencies
├── faiss_utils/
│   ├── __init__.py
│   └── faiss_index.py               # FAISS index wrapper
├── utils/
│   ├── preprocess.py                # Image preprocessing utilities
│   ├── tensorrt_infer.py           # TensorRT inference engine
│   └── export_engine.py            # Model export utilities
└── tensorrt/                        # TensorRT engine files
    └── resnet50_fp16.engine
```

## Usage

### 1. Building Indexes

#### Full Image Index
```bash
python build_index.py
```

#### Bbox-aware Index
```bash
python build_index_with_crop_img.py
```

### 2. Performing Searches

#### Basic Search
```python
from search import search
from faiss_utils.faiss_index import FaissIndexer

# Load index
indexer = FaissIndexer(feature_dim, "tensorrt/AMEX.index")

# Search for similar images
D, I = search(query_image_path, image_paths, indexer, index_path, topk=5)
```

#### Search for Problematic Images
```bash
python search_bad_imgs.py
```

#### Enhanced Icon Search
```bash
python search_bad_imgs_with_icon.py
```

### 3. Data Processing

#### Copy Matched Data
```bash
python copy_matched_icon_data_to_new_path.py
```

## Configuration

### Dataset Configuration

Modify the `index_data_list` in the respective files to configure your datasets:

```python
index_data_list = [
    {
        "index_name": "tensorrt/dataset1.index", 
        "data_dir": "./data/dataset1/images",
        "json_path": "./data/dataset1/metadata/hf_train.json"  # For bbox-aware
    },
    # Add more datasets...
]
```

### Model Configuration

Key parameters in `build_index.py`:

```python
gpu_id = 4                    # GPU device ID
batch_size = 128              # Batch size for processing
feature_dim = 2048            # ResNet50 feature dimension
input_shape = (112, 112)      # Input image size
engine_path = 'tensorrt/resnet50_fp16.engine'  # TensorRT engine path
```

## Model Export

To export ResNet50 to TensorRT format:

```bash
python utils/export_engine.py
```

This will:
1. Export PyTorch ResNet50 to ONNX format
2. Convert ONNX to TensorRT engine with FP16 optimization
3. Save the engine file for inference

## Performance Optimization

### GPU Configuration
- Set `CUDA_VISIBLE_DEVICES` to specify available GPUs
- Adjust `batch_size` based on GPU memory
- Use FP16 precision for faster inference

### Index Optimization
- Use FAISS-GPU for faster search
- Consider using IVF indices for very large datasets
- Implement batch processing for large-scale operations

## Data Format

### Image Data
- Supported formats: JPG, JPEG, PNG, BMP
- Recommended size: 112x112 pixels
- RGB color space

### Annotation Data (for bbox-aware search)
```json
[
  {
    "img_url": "path/to/image.png",
    "img_size": [width, height],
    "element": [
      {
        "bbox": [x1, y1, x2, y2],
        "instruction": "click button"
      }
    ]
  }
]
```

## API Reference

### Core Functions

#### `search(query_img, image_paths, indexer, index_path, topk=5)`
Performs similarity search on a query image.

**Parameters:**
- `query_img`: Path to query image
- `image_paths`: List of indexed image paths
- `indexer`: FAISS indexer instance
- `index_path`: Path to FAISS index file
- `topk`: Number of top results to return

**Returns:**
- `D`: Distance array
- `I`: Index array

#### `build_index(index_path, data_dir)`
Builds FAISS index from image directory.

**Parameters:**
- `index_path`: Output path for FAISS index
- `data_dir`: Directory containing images

### Utility Classes

#### `FaissIndexer`
FAISS index wrapper with convenient methods.

#### `TRTInference`
TensorRT inference engine for ResNet50.

#### `preprocess_image1(image_path, input_shape, gpu_id)`
Preprocesses image for model input.

## Troubleshooting

### Common Issues

1. **CUDA Out of Memory**
   - Reduce batch size
   - Use smaller input images
   - Clear GPU cache between operations

2. **TensorRT Engine Loading Failed**
   - Verify TensorRT installation
   - Check engine file path
   - Ensure compatible CUDA version

3. **FAISS Index Errors**
   - Install FAISS-GPU instead of FAISS-CPU
   - Check index file integrity
   - Verify feature dimensions match

### Performance Tips

1. **Batch Processing**: Process images in batches for better GPU utilization
2. **Memory Management**: Use context managers for CUDA operations
3. **Index Optimization**: Choose appropriate FAISS index type for your dataset size

## Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests if applicable
5. Submit a pull request

## License

This project is licensed under the MIT License - see the LICENSE file for details.

## Acknowledgments

- NVIDIA for TensorRT optimization
- Facebook AI Research for FAISS
- PyTorch team for the deep learning framework
- ResNet50 architecture by Microsoft Research
