import faiss
import numpy as np
import os

class FaissIndexer:

    def __init__(self, dim: int, index_path: str = None):
        self.dim = dim
        self.index_path = index_path

        try:
            self.index = faiss.IndexFlatL2(dim)
        except AttributeError:
            raise ImportError("Current faiss package does not support IndexFlatL2 and index_factory, please install official faiss or faiss-cpu package")
      
        if index_path and os.path.exists(index_path):
            self.load(index_path)

    def add(self, feats: np.ndarray):
        self.index.add(feats)

    def search(self, query: np.ndarray, topk: int = 5):
        D, I = self.index.search(query, topk)
        return D, I

    def save(self, path: str = None):
        if path is None:
            path = self.index_path
        faiss.write_index(self.index, path)

    def load(self, path: str = None):
        if path is None:
            path = self.index_path
        self.index = faiss.read_index(path)
    def get_all_vectors(self):
        """
        Return all feature vectors as numpy array, shape=(N, dim)
        """
        if hasattr(self, 'index') and self.index is not None:
            try:
                return self.index.reconstruct_n(0, self.index.ntotal)
            except Exception:
                # Compatible with some faiss versions
                try:
                    return self.index.xb.copy() if hasattr(self.index, 'xb') else np.array([self.index.reconstruct(i) for i in range(self.index.ntotal)])
                except Exception as e:
                    print(f"[ERROR] get_all_vectors failed: {e}")
                    return None
        return None
