
import torch
import torchvision.transforms as T
from PIL import Image
import numpy as np
from typing import List
import cv2
def preprocess_image1(image_path: str, input_shape=(224, 224), gpu_id: int = 0) -> np.ndarray:
    """
    Read image and preprocess, return normalized numpy array.
    """
    try:
        img = cv2.imread(image_path)
        img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
        img = cv2.resize(img, input_shape)
        img = img.astype(np.float32) / 255.0
        # ImageNet mean and variance
        mean = np.array([0.485, 0.456, 0.406], dtype=np.float32)
        std = np.array([0.229, 0.224, 0.225], dtype=np.float32)
        img = (img - mean) / std
        img = np.transpose(img, (2, 0, 1))  # CHW
        return img
    except Exception as e:
        print(f"[ERROR] preprocess_image1 failed: {image_path}, error: {e}")
        return None
def preprocess_image2(image_path: str, input_shape=(224, 224), gpu_id: int = 0) -> np.ndarray:
    """
    Read image with torch on GPU and preprocess, return normalized numpy array.
    """
    transform = T.Compose([
        T.Resize(input_shape),
        T.ToTensor(),
        T.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])
    ])
    device = torch.device(f'cuda:{gpu_id}')
    img = Image.open(image_path).convert('RGB')
    tensor = transform(img).to(device)
    arr = tensor.cpu().numpy().astype(np.float32)
    arr = np.ascontiguousarray(arr)
    return arr

def preprocess_batch(image_paths: List[str], input_shape=(224, 224), gpu_id: int = 0) -> np.ndarray:
    batch = [preprocess_image2(p, input_shape, gpu_id) for p in image_paths]
    return np.stack(batch)

