#!/usr/bin/env python
# Copyright (c) OpenMMLab. All rights reserved.
import functools as func
import glob
import os.path as osp
import re

import numpy as np

url_prefix = 'https://github.com/open-mmlab/mmsegmentation/blob/master/'

files = sorted(glob.glob('../../configs/*/README.md'))

stats = []
titles = []
num_ckpts = 0

for f in files:
    url = osp.dirname(f.replace('../../', url_prefix))

    with open(f) as content_file:
        content = content_file.read()

    title = content.split('\n')[0].replace('#', '').strip()
    ckpts = {
        x.lower().strip()
        for x in re.findall(r'https?://download.*\.pth', content)
        if 'mmsegmentation' in x
    }
    if len(ckpts) == 0:
        continue

    _papertype = [
        x for x in re.findall(r'<!--\s*\[([A-Z]*?)\]\s*-->', content)
    ]
    assert len(_papertype) > 0
    papertype = _papertype[0]

    paper = {(papertype, title)}

    titles.append(title)
    num_ckpts += len(ckpts)
    statsmsg = f"""
\t* [{papertype}] [{title}]({url}) ({len(ckpts)} ckpts)
"""
    stats.append((paper, ckpts, statsmsg))

allpapers = func.reduce(lambda a, b: a.union(b), [p for p, _, _ in stats])
msglist = '\n'.join(x for _, _, x in stats)

papertypes, papercounts = np.unique([t for t, _ in allpapers],
                                    return_counts=True)
countstr = '\n'.join(
    [f'   - {t}: {c}' for t, c in zip(papertypes, papercounts)])

modelzoo = f"""
# 模型库统计数据

* 论文数量: {len(set(titles))}
{countstr}

* 模型数量: {num_ckpts}
{msglist}
"""

with open('modelzoo_statistics.md', 'w') as f:
    f.write(modelzoo)
