"""
For DMControl and Gym MuJoCo, there are no human-normalized scores. 
Therefore, we normalize the scores based on the performance the expert agent reaches at the end of training. 
--> Data normalized

Random and expert refernence scores for D4RL are available here: https://github.com/Farama-Foundation/D4RL/blob/master/d4rl/infos.py

"""
import math
import dmc2gym
import numpy as np
import pandas as pd
from .env_names import DM_CONTROL_ENVS, ATARI_NAME_TO_ENVID


# Task: score-tuple dictionary. Each score tuple contains
#  0: score random (float) and 1: mean scores in the datasets (float).
ENVID_TO_DNS = {
    'acrobot-swingup': (8.351, 4.877), 
    'ball_in_cup-catch': (0.0, 926.719), 
    'cartpole-balance': (350.391, 938.506),
    'cartpole-swingup': (27.414, 766.15),
    'cheetah-run': (3.207, 324.045),
    'finger-spin': (0.2, 834.629),
    'finger-turn_easy': (57.8, 800.645),
    'finger-turn_hard': (40.6, 676.144),
    'fish-swim': (67.675, 78.212),
    'fish-upright': (229.406, 547.962),
    'hopper-hop': (0.076, 62.794),
    'hopper-stand': (1.296, 266.783), 
    'humanoid-run': (0.741, 0.794), 
    'humanoid-stand': (4.327, 5.053),
    'humanoid-walk': (0.913, 1.194), 
    'manipulator-bring_ball': (0.0, 0.429), 
    'manipulator-insert_ball': (0.0, 43.307), 
    'manipulator-insert_peg': (0.235, 78.477), 
    'pendulum-swingup': (0.0, 614.491), 
    'point_mass-easy': (1.341, 779.273),
    'reacher-easy': (33.0, 849.241), 
    'reacher-hard': (8.0, 779.947), 
    'swimmer-swimmer15': (78.817, 152.297), 
    'swimmer-swimmer6': (229.834, 167.082),
    'walker-run': (23.427, 344.794),
    'walker-stand': (134.701, 816.322),
    'walker-walk': (30.193, 773.174),
    "HalfCheetah-v3":(-280.178953, 12135.0), 
    "Walker2d-v3": (1.629008, 4592.3),
    "Hopper-v3": (-20.272305, 3234.3),
    "HalfCheetah-v2":(-280.178953, 12135.0), 
    "Walker2d-v2": (1.629008, 4592.3), 
    "Hopper-v2": (-20.272305, 3234.3),
    # extracted from 25M data
    "bigfish":   (0.0, 5.9107),
    "bossfight": (0.0, 2.179),
    "caveflyer": (0.0, 7.6341),
    "chaser":    (0.0, 3.4349),
    "climber":   (0.0, 9.1516),
    "coinrun":   (0.0, 9.6781),
    "dodgeball": (0.0, 3.1873),
    "fruitbot":  (-7.0, 16.9643),
    "heist":     (0.0, 7.9555),
    "jumper":    (0.0, 8.7396),
    "leaper":    (0.0, 4.9065),
    "maze":      (0.0, 9.4536),
    "miner":     (0.0, 11.6814),
    "ninja":     (0.0, 7.7674),
    "plunder":   (0.0, 4.9095),
    "starpilot": (0.0, 17.3367),
    # metaworld
    "assembly-v2": (15.0363, 1285.642),
    "button-press-v2": (46.4804, 1604.5227),
    "disassemble-v2": (0.0, 1536.4565),
    "plate-slide-side-v2": (13.2441, 1748.3042),
    "door-lock-v2": (177.2109, 1831.5374),
    "door-unlock-v2": (123.2161, 1794.7216),
    "pick-place-v2": (0.3011, 1300.6919),
    "drawer-close-v2": (0.0, 1880.0),
    "coffee-pull-v2": (4.6319, 1475.0729),
    "handle-pull-v2": (2.7437, 1759.0701),
    "peg-insert-side-v2": (0.6391, 1695.9866),
    "door-close-v2": (5.4191, 1587.8),
    "button-press-topdown-v2": (35.1767, 1384.7),
    "lever-pull-v2": (33.9352, 1677.197),
    "reach-v2": (147.103, 1905.1068),
    "door-open-v2": (83.5917, 1608.8119),
    "sweep-v2": (2.8682, 1560.8198),
    "box-close-v2": (0.0, 1127.9011),
    "button-press-wall-v2": (8.2484, 1614.5718),
    "basketball-v2": (0.2175, 1597.0182),
    "soccer-v2": (0.7333, 1706.2247),
    "handle-press-v2": (9.0016, 1929.8344),
    "coffee-button-v2": (40.8464, 1698.3198),
    "faucet-open-v2": (327.8281, 1783.7488),
    "plate-slide-v2": (73.5299, 1713.0806),
    "reach-wall-v2": (141.9194, 1863.1053),
    "handle-pull-side-v2": (0.9445, 1710.8367),
    "drawer-open-v2": (179.8318, 1762.8101),
    "dial-turn-v2": (0.0004, 1887.3281),
    "sweep-into-v2": (4.0911, 1815.943),
    "bin-picking-v2": (1.7012, 1308.5978),
    "coffee-push-v2": (2.2135, 1694.6849),
    "pick-out-of-hole-v2": (0.7432, 1588.1172),
    "button-press-topdown-wall-v2": (35.1767, 1388.5978),
    "plate-slide-back-side-v2": (25.7285, 1833.703),
    "plate-slide-back-v2": (29.671, 1806.1564),
    "stick-push-v2": (1.0867, 1629.2634),
    "hand-insert-v2": (0.6889, 1789.0469),
    "window-open-v2": (40.516, 1717.4207),
    "pick-place-wall-v2": (0.0, 1654.6439),
    "hammer-v2": (31.6972, 1724.8417),
    "peg-unplug-side-v2": (1.6908, 1619.8203),
    "window-close-v2": (30.2317, 1559.3096),
    "faucet-close-v2": (410.239, 1798.5878),
    "handle-press-side-v2": (22.6648, 1897.76),
    "push-back-v2": (0.442, 1564.7234),
    "push-v2": (2.0408, 1791.2244),
    "push-wall-v2": (2.0338, 1746.7048),
    "stick-pull-v2": (0.6954, 1551.3679),
    "shelf-place-v2": (0.0, 1513.3306),
    # atari --> use HN scores --> rewards are clipped in dataset
    "alien": (227.8, 7127.7),
    "amidar": (5.8, 1719.5),
    "assault": (222.4, 742.0),
    "asterix": (210.0, 8503.3),
    "asteroids": (719.1, 47388.7),
    "atlantis": (12850.0, 29028.1),
    "bank-heist": (14.2, 753.1),
    "battle-zone": (2360.0, 37187.5),
    "beam-rider": (363.9, 16926.5),
    "berzerk": (123.7, 2630.4),
    "bowling": (23.1, 160.7),
    "boxing": (0.1, 12.1),
    "breakout": (1.7, 30.5),
    "centipede": (2090.9, 12017.0),
    "chopper-command": (811.0, 7387.8),
    "crazy-climber": (10780.5, 35829.4),
    "defender": (2874.5, 18688.9),
    "demon-attack": (152.1, 1971.0),
    "double-dunk": (-18.6, -16.4),
    "enduro": (0.0, 860.5),
    "fishing-derby": (-91.7, -38.7),
    "freeway": (0.0, 29.6),
    "frostbite": (65.2, 4334.7),
    "gopher": (257.6, 2412.5),
    "gravitar": (173.0, 3351.4),
    "hero": (1027.0, 30826.4),
    "ice-hockey": (-11.2, 0.9),
    "jamesbond": (29.0, 302.8),
    "kangaroo": (52.0, 3035.0),
    "krull": (1598.0, 2665.5),
    "kung-fu-master": (258.5, 22736.3),
    "montezuma-revenge": (0.0, 4753.3),
    "ms-pacman": (307.3, 6951.6),
    "name-this-game": (2292.3, 8049.0),
    "phoenix": (761.4, 7242.6),
    "pitfall": (-229.4, 6463.7),
    "pong": (-20.7, 14.6),
    "private-eye": (24.9, 69571.3),
    "qbert": (163.9, 13455.0),
    "riverraid": (1338.5, 17118.0),
    "road-runner": (11.5, 7845.0),
    "robotank": (2.2, 11.9),
    "seaquest": (68.4, 42054.7),
    "skiing": (-17098.1, -4336.9),
    "solaris": (1236.3, 12326.7),
    "space-invaders": (148.0, 1668.7),
    "star-gunner": (664.0, 10250.0),
    "surround": (-10.0, 6.5),
    "tennis": (-23.8, -8.3),
    "time-pilot": (3568.0, 5229.2),
    "tutankham": (11.4, 167.6),
    "up-n-down": (533.4, 11693.2),
    "venture": (0.0, 1187.5),
    # Note the random agent score on Video Pinball is sometimes greater than the
    # human score under other evaluation methods.
    "video-pinball": (16256.9, 17667.9),
    "wizard-of-wor": (563.5, 4756.5),
    "yars-revenge": (3092.9, 54576.9),
    "zaxxon": (32.5, 9173.3),
    # composuite --> min/max in dataset
    "Panda_Hollowbox_ObjectDoor_Push": (24.7741, 322.4837),
    "Panda_Box_ObjectDoor_PickPlace": (17.2628, 329.0957),
    "Panda_Plate_ObjectDoor_Push": (19.2525, 307.8281),
    "Panda_Plate_GoalWall_Push": (9.685, 348.3908),
    "Kinova3_Box_ObjectWall_Trashcan": (7.6667, 395.7307),
    "Panda_Plate_GoalWall_PickPlace": (14.482, 377.3351),
    "Jaco_Dumbbell_None_PickPlace": (56.2307, 355.6125),
    "Kinova3_Dumbbell_None_Push": (20.8669, 405.8568),
    "Panda_Plate_GoalWall_Trashcan": (13.4391, 367.3199),
    "Jaco_Hollowbox_ObjectDoor_PickPlace": (11.9709, 356.0208),
    "Jaco_Box_ObjectDoor_PickPlace": (2.0693, 366.4434),
    "Panda_Dumbbell_None_Trashcan": (17.7564, 372.779),
    "IIWA_Box_None_Push": (38.436, 423.2112),
    "Jaco_Plate_GoalWall_Push": (12.5549, 327.3481),
    "Jaco_Plate_ObjectWall_PickPlace": (10.4849, 355.2842),
    "IIWA_Hollowbox_ObjectDoor_Push": (24.2423, 381.1122),
    "Jaco_Plate_ObjectWall_Shelf": (8.7829, 319.127),
    "Panda_Box_None_Trashcan": (18.9648, 392.7072),
    "Panda_Hollowbox_GoalWall_PickPlace": (12.6573, 389.1634),
    "Kinova3_Box_None_PickPlace": (38.4416, 425.4009),
    "Jaco_Box_ObjectDoor_Shelf": (29.5737, 330.9081),
    "IIWA_Box_GoalWall_PickPlace": (9.1711, 417.9734),
    "Kinova3_Dumbbell_None_Trashcan": (42.9124, 390.6068),
    "Kinova3_Dumbbell_ObjectWall_Trashcan": (18.9259, 373.2451),
    "IIWA_Plate_None_Trashcan": (5.6109, 397.7719),
    "Jaco_Box_ObjectDoor_Push": (39.5665, 361.8412),
    "IIWA_Box_None_Trashcan": (19.4895, 414.214),
    "Kinova3_Box_ObjectDoor_PickPlace": (14.6282, 408.9602),
    "IIWA_Plate_None_Shelf": (17.6037, 374.8415),
    "Panda_Hollowbox_ObjectDoor_Shelf": (18.2482, 312.36),
    "Kinova3_Hollowbox_GoalWall_Push": (18.8628, 369.8759),
    "Jaco_Box_None_Push": (34.9191, 386.1124),
    "Panda_Dumbbell_ObjectDoor_PickPlace": (21.0527, 303.0532),
    "Jaco_Box_GoalWall_Trashcan": (11.7735, 369.816),
    "Jaco_Hollowbox_None_PickPlace": (28.6506, 378.2476),
    "Jaco_Box_ObjectWall_Shelf": (8.4777, 330.0855),
    "Jaco_Dumbbell_None_Push": (26.4615, 356.8932),
    "Panda_Dumbbell_ObjectDoor_Push": (17.5614, 309.5276),
    "Jaco_Dumbbell_ObjectDoor_Trashcan": (24.3902, 325.079),
    "Jaco_Hollowbox_None_Trashcan": (11.1675, 362.976),
    "Jaco_Hollowbox_ObjectWall_Push": (19.5258, 360.565),
    "Jaco_Hollowbox_ObjectWall_Trashcan": (9.2933, 342.2538),
    "Jaco_Box_GoalWall_Push": (22.5953, 321.1584),
    "Panda_Plate_None_PickPlace": (19.51, 380.8439),
    "IIWA_Hollowbox_GoalWall_PickPlace": (6.3764, 419.541),
    "Panda_Box_None_Push": (35.4139, 397.4014),
    "Panda_Hollowbox_ObjectWall_Shelf": (7.3328, 289.9606),
    "IIWA_Plate_ObjectWall_Push": (12.883, 395.6757),
    "IIWA_Dumbbell_ObjectDoor_Shelf": (44.5049, 332.6358),
    "Panda_Dumbbell_ObjectWall_PickPlace": (28.006, 313.7836),
    "Panda_Hollowbox_ObjectWall_Push": (18.8042, 330.8746),
    "Panda_Plate_ObjectDoor_Trashcan": (12.1427, 293.2389),
    "Panda_Plate_None_Trashcan": (12.8828, 367.6216),
    "Panda_Dumbbell_None_PickPlace": (31.5137, 373.9502),
    "Kinova3_Dumbbell_GoalWall_Trashcan": (32.6547, 386.5341),
    "Panda_Hollowbox_None_Shelf": (15.2793, 349.8202),
    "Panda_Hollowbox_None_Push": (21.0601, 399.9587),
    "IIWA_Hollowbox_ObjectWall_PickPlace": (7.4666, 406.4884),
    "Kinova3_Dumbbell_ObjectWall_PickPlace": (11.0238, 388.9163),
    "Panda_Dumbbell_GoalWall_Trashcan": (22.7508, 357.7439),
    "Jaco_Hollowbox_ObjectDoor_Shelf": (11.1222, 318.0287),
    "IIWA_Plate_ObjectDoor_Push": (11.5567, 373.6541),
    "Jaco_Plate_GoalWall_Shelf": (14.4472, 334.1012),
    "Panda_Dumbbell_ObjectDoor_Trashcan": (22.4707, 284.3753),
    "IIWA_Hollowbox_ObjectDoor_PickPlace": (12.4441, 386.7481),
    "Panda_Box_GoalWall_Shelf": (32.9995, 352.3642),
    "IIWA_Hollowbox_ObjectWall_Push": (37.1698, 400.5991),
    "IIWA_Plate_GoalWall_PickPlace": (17.8362, 408.5447),
    "Kinova3_Plate_ObjectWall_Trashcan": (13.1361, 382.8602),
    "Panda_Plate_GoalWall_Shelf": (10.0223, 340.3288),
    "Panda_Plate_ObjectWall_Trashcan": (11.0588, 303.8733),
    "Jaco_Dumbbell_None_Trashcan": (30.3643, 341.4949),
    "Panda_Plate_ObjectWall_Shelf": (16.4306, 277.7495),
    "Jaco_Plate_ObjectDoor_Shelf": (13.0975, 323.689),
    "IIWA_Plate_ObjectWall_Trashcan": (16.4352, 379.8558),
    "Kinova3_Hollowbox_ObjectWall_Push": (27.6849, 397.4564),
    "Jaco_Dumbbell_None_Shelf": (49.8153, 325.7801),
    "Kinova3_Box_None_Push": (30.0559, 428.0315),
    "Kinova3_Plate_ObjectWall_Shelf": (13.871, 370.975),
    "IIWA_Plate_ObjectDoor_Shelf": (5.4965, 349.4955),
    "Panda_Box_ObjectDoor_Shelf": (11.8494, 300.808),
    "IIWA_Plate_ObjectDoor_PickPlace": (6.6643, 381.6441),
    "Kinova3_Plate_GoalWall_Shelf": (12.0883, 373.31),
    "Panda_Dumbbell_None_Push": (21.1857, 373.3488),
    "Kinova3_Hollowbox_ObjectWall_Trashcan": (8.393, 379.6972),
    "Jaco_Plate_ObjectDoor_Trashcan": (7.292, 337.2096),
    "Panda_Hollowbox_ObjectWall_PickPlace": (8.4047, 345.2379),
    "Panda_Hollowbox_None_Trashcan": (7.6515, 382.1819),
    "Panda_Box_ObjectWall_Push": (25.8325, 330.4575),
    "Jaco_Box_None_Shelf": (13.5463, 348.9094),
    "Kinova3_Plate_GoalWall_PickPlace": (23.7958, 406.7204),
    "Panda_Box_ObjectWall_PickPlace": (1.4694, 338.1854),
    "Panda_Box_GoalWall_Trashcan": (14.9617, 392.5051),
    "Kinova3_Hollowbox_GoalWall_PickPlace": (22.4748, 413.5353),
    "IIWA_Plate_None_PickPlace": (16.1579, 414.3734),
    "IIWA_Box_GoalWall_Trashcan": (10.9236, 411.6396),
    "Panda_Dumbbell_GoalWall_Shelf": (25.9432, 324.7186),
    "IIWA_Dumbbell_ObjectDoor_Push": (42.0848, 372.8149),
    "Panda_Box_ObjectWall_Shelf": (30.8648, 297.7375),
    "IIWA_Plate_GoalWall_Trashcan": (11.8227, 400.2317),
    "IIWA_Plate_ObjectWall_Shelf": (11.4958, 351.4135),
    "Jaco_Box_None_Trashcan": (11.1733, 377.2025),
    "Jaco_Plate_GoalWall_Trashcan": (18.3085, 347.8802),
    "IIWA_Dumbbell_GoalWall_Trashcan": (31.6533, 384.8889),
    "Kinova3_Plate_ObjectWall_PickPlace": (6.8484, 398.0574),
    "Kinova3_Plate_None_Push": (32.003, 407.4045),
    "Panda_Hollowbox_ObjectWall_Trashcan": (12.1934, 331.1017),
    "Panda_Dumbbell_ObjectWall_Shelf": (20.7777, 280.836),
    "Kinova3_Hollowbox_GoalWall_Shelf": (20.8917, 404.2996),
    "IIWA_Plate_GoalWall_Shelf": (12.235, 372.4099),
    "Panda_Dumbbell_None_Shelf": (50.2999, 329.6797),
    "Jaco_Hollowbox_GoalWall_PickPlace": (13.9411, 362.3539),
    "Jaco_Dumbbell_ObjectWall_Trashcan": (30.5, 325.5367),
    "IIWA_Box_None_PickPlace": (34.2943, 422.6896),
    "IIWA_Box_ObjectWall_Shelf": (10.3803, 369.3631),
    "IIWA_Box_GoalWall_Push": (32.7633, 372.0507),
    "Panda_Box_ObjectDoor_Trashcan": (9.1216, 315.0218),
    "Panda_Hollowbox_GoalWall_Trashcan": (10.0555, 378.0457),
    "Jaco_Box_GoalWall_PickPlace": (10.4602, 372.9176),
    "Jaco_Dumbbell_GoalWall_PickPlace": (107.952, 348.9397),
    "Kinova3_Dumbbell_GoalWall_Shelf": (28.4337, 355.2201),
    "Kinova3_Box_GoalWall_Trashcan": (8.3963, 409.4071),
    "IIWA_Plate_ObjectWall_PickPlace": (12.6105, 396.0394),
    "Panda_Plate_None_Shelf": (11.7839, 341.5723),
    "Jaco_Box_ObjectWall_Trashcan": (24.9974, 350.9608),
    "Jaco_Box_ObjectWall_PickPlace": (8.8292, 371.855),
    "Kinova3_Plate_ObjectDoor_Push": (8.7078, 393.791),
    "Kinova3_Box_None_Shelf": (18.2475, 389.364),
    "Kinova3_Plate_ObjectWall_Push": (26.0772, 394.4429),
    "IIWA_Dumbbell_ObjectWall_Trashcan": (31.0994, 365.8652),
    "Kinova3_Box_GoalWall_Push": (24.6329, 373.9586),
    "IIWA_Box_ObjectWall_Trashcan": (8.8909, 394.6256),
    "IIWA_Box_ObjectDoor_Trashcan": (10.077, 377.3615),
    "Jaco_Hollowbox_None_Shelf": (15.0291, 346.1891),
    "Kinova3_Dumbbell_GoalWall_PickPlace": (19.3837, 393.442),
    "Kinova3_Dumbbell_None_PickPlace": (48.5357, 401.8801),
    "Panda_Hollowbox_GoalWall_Push": (9.8277, 360.0789),
    "IIWA_Hollowbox_ObjectDoor_Shelf": (16.4937, 343.6018),
    "Kinova3_Dumbbell_ObjectDoor_Trashcan": (38.2474, 372.15),
    "IIWA_Hollowbox_GoalWall_Shelf": (19.8792, 403.9416),
    "Kinova3_Plate_None_Trashcan": (7.4133, 395.1252),
    "IIWA_Dumbbell_None_Push": (31.3429, 403.705),
    "Jaco_Hollowbox_ObjectDoor_Trashcan": (17.7145, 337.9608),
    "IIWA_Dumbbell_GoalWall_Shelf": (13.7079, 358.069),
    "Jaco_Dumbbell_ObjectDoor_PickPlace": (21.2415, 348.8272),
    "IIWA_Dumbbell_None_PickPlace": (19.8469, 408.9983),
    "IIWA_Hollowbox_ObjectWall_Shelf": (22.6392, 364.1277),
    "Jaco_Box_ObjectWall_Push": (35.5941, 369.6819),
    "Kinova3_Dumbbell_ObjectDoor_PickPlace": (14.0273, 383.0844),
    "Kinova3_Hollowbox_ObjectDoor_Shelf": (5.5156, 350.2031),
    "Kinova3_Dumbbell_GoalWall_Push": (13.1288, 355.5746),
    "IIWA_Hollowbox_None_Shelf": (17.2876, 403.8859),
    "Jaco_Plate_None_Push": (9.8736, 367.8203),
    "Kinova3_Dumbbell_ObjectDoor_Shelf": (7.4872, 339.1107),
    "Kinova3_Plate_None_PickPlace": (20.5777, 411.8783),
    "Kinova3_Box_ObjectDoor_Push": (7.399, 396.5578),
    "Kinova3_Dumbbell_ObjectWall_Push": (19.5603, 386.0225),
    "Kinova3_Box_ObjectDoor_Shelf": (13.3729, 366.5801),
    "IIWA_Dumbbell_None_Trashcan": (31.5699, 389.3736),
    "Jaco_Hollowbox_ObjectDoor_Push": (18.5165, 353.3318),
    "Kinova3_Box_None_Trashcan": (6.9692, 415.2254),
    "Jaco_Plate_GoalWall_PickPlace": (20.4602, 358.3443),
    "Kinova3_Hollowbox_None_Push": (20.102, 418.2462),
    "IIWA_Box_ObjectWall_Push": (40.764, 402.1152),
    "Panda_Box_GoalWall_PickPlace": (11.3912, 391.9168),
    "IIWA_Dumbbell_ObjectWall_Push": (9.4897, 389.1574),
    "Panda_Box_ObjectWall_Trashcan": (12.8595, 323.1698),
    "Jaco_Hollowbox_ObjectWall_Shelf": (9.6975, 321.8891),
    "Jaco_Plate_ObjectWall_Trashcan": (8.051, 334.6109),
    "Kinova3_Hollowbox_ObjectDoor_PickPlace": (10.6141, 396.7687),
    "Kinova3_Hollowbox_ObjectDoor_Push": (4.3734, 398.4861),
    "IIWA_Plate_GoalWall_Push": (10.9814, 376.582),
    "Kinova3_Plate_GoalWall_Push": (15.4011, 369.438),
    "Kinova3_Plate_ObjectDoor_Trashcan": (5.8734, 378.2037),
    "Jaco_Box_GoalWall_Shelf": (28.1746, 348.0137),
    "Kinova3_Plate_None_Shelf": (19.3038, 376.0602),
    "Jaco_Box_ObjectDoor_Trashcan": (10.2635, 348.2585),
    "IIWA_Plate_None_Push": (8.5393, 407.4024),
    "Jaco_Box_None_PickPlace": (10.6097, 380.3599),
    "Kinova3_Plate_ObjectDoor_PickPlace": (10.8501, 396.2977),
    "IIWA_Box_ObjectDoor_Shelf": (12.3437, 355.1942),
    "Panda_Box_None_Shelf": (12.817, 357.7363),
    "IIWA_Dumbbell_ObjectWall_Shelf": (44.0715, 341.2852),
    "Kinova3_Hollowbox_None_PickPlace": (12.4409, 422.8646),
    "Kinova3_Hollowbox_ObjectWall_Shelf": (30.8403, 375.4196),
    "Jaco_Dumbbell_GoalWall_Push": (15.047, 295.2301),
    "Kinova3_Plate_GoalWall_Trashcan": (6.1659, 399.722),
    "Panda_Box_GoalWall_Push": (34.7895, 300.7138),
    "Kinova3_Box_ObjectWall_PickPlace": (8.3143, 413.1352),
    "Panda_Plate_ObjectDoor_PickPlace": (17.262, 314.5126),
    "Jaco_Hollowbox_GoalWall_Shelf": (7.6225, 339.0533),
    "Panda_Box_None_PickPlace": (10.1489, 402.7562),
    "Kinova3_Dumbbell_None_Shelf": (37.2735, 363.8753),
    "Jaco_Plate_ObjectDoor_PickPlace": (19.644, 353.0497),
    "Panda_Hollowbox_GoalWall_Shelf": (21.3369, 348.1734),
    "Jaco_Hollowbox_GoalWall_Trashcan": (29.6536, 355.6227),
    "Jaco_Hollowbox_GoalWall_Push": (16.6002, 322.2183),
    "Jaco_Dumbbell_GoalWall_Shelf": (58.5558, 322.3699),
    "Kinova3_Box_ObjectWall_Push": (1.2927, 405.5613),
    "Jaco_Dumbbell_ObjectDoor_Shelf": (39.8351, 304.0632),
    "IIWA_Hollowbox_ObjectWall_Trashcan": (13.383, 390.5933),
    "IIWA_Box_ObjectDoor_PickPlace": (43.9642, 392.528),
    "Jaco_Plate_None_PickPlace": (17.8849, 366.7162),
    "Jaco_Hollowbox_ObjectWall_PickPlace": (8.5119, 363.9757),
    "IIWA_Hollowbox_None_PickPlace": (14.9898, 422.507),
    "IIWA_Dumbbell_ObjectWall_PickPlace": (10.683, 391.7573),
    "Kinova3_Dumbbell_ObjectDoor_Push": (36.5199, 383.6134),
    "IIWA_Hollowbox_None_Trashcan": (15.6829, 413.8112),
    "IIWA_Box_ObjectWall_PickPlace": (12.1884, 407.7422),
    "Jaco_Dumbbell_GoalWall_Trashcan": (29.7935, 339.696),
    "IIWA_Box_GoalWall_Shelf": (31.2831, 388.9835),
    "IIWA_Hollowbox_None_Push": (19.8047, 420.6052),
    "IIWA_Box_None_Shelf": (22.2564, 392.7056),
    "Kinova3_Hollowbox_GoalWall_Trashcan": (10.6612, 396.6298),
    "Panda_Dumbbell_ObjectWall_Push": (19.7928, 320.4766),
    "IIWA_Dumbbell_ObjectDoor_PickPlace": (6.4959, 375.7194),
    "IIWA_Hollowbox_GoalWall_Push": (17.0836, 378.3529),
    "Jaco_Plate_ObjectDoor_Push": (13.168, 341.8283),
    "Panda_Dumbbell_ObjectWall_Trashcan": (19.4364, 294.1954),
    "Kinova3_Box_GoalWall_PickPlace": (17.8266, 415.1603),
    "Panda_Dumbbell_GoalWall_PickPlace": (22.6342, 360.2449),
    "Panda_Box_ObjectDoor_Push": (8.7999, 328.241),
    "Jaco_Dumbbell_ObjectDoor_Push": (10.2356, 340.2007),
    "Jaco_Dumbbell_ObjectWall_Push": (14.5793, 337.6736),
    "Panda_Hollowbox_ObjectDoor_PickPlace": (6.6202, 329.384),
    "IIWA_Dumbbell_GoalWall_Push": (17.2815, 358.6089),
    "Jaco_Plate_ObjectWall_Push": (15.6346, 348.5868),
    "IIWA_Hollowbox_GoalWall_Trashcan": (10.0154, 403.7551),
    "IIWA_Dumbbell_GoalWall_PickPlace": (13.7382, 398.5261),
    "Kinova3_Hollowbox_ObjectDoor_Trashcan": (12.6474, 379.065),
    "Jaco_Hollowbox_None_Push": (16.4941, 381.7607),
    "Kinova3_Box_ObjectWall_Shelf": (7.9664, 366.0194),
    "IIWA_Dumbbell_None_Shelf": (17.9913, 360.1213),
    "Kinova3_Hollowbox_ObjectWall_PickPlace": (17.7919, 402.5087),
    "Jaco_Plate_None_Trashcan": (12.4709, 362.5799),
    "Jaco_Dumbbell_ObjectWall_Shelf": (54.9856, 296.5891),
    "Panda_Plate_ObjectWall_PickPlace": (14.8556, 318.9778),
    "Kinova3_Plate_ObjectDoor_Shelf": (17.9693, 363.9405),
    "Jaco_Dumbbell_ObjectWall_PickPlace": (11.0959, 347.2706),
    "Kinova3_Box_ObjectDoor_Trashcan": (10.883, 389.944),
    "Kinova3_Box_GoalWall_Shelf": (13.6893, 385.2776),
    "IIWA_Hollowbox_ObjectDoor_Trashcan": (13.0273, 369.7449),
    "Kinova3_Dumbbell_ObjectWall_Shelf": (8.6434, 336.2771),
    "Jaco_Plate_None_Shelf": (11.0834, 342.9441),
    "IIWA_Plate_ObjectDoor_Trashcan": (11.1007, 366.6993),
    "IIWA_Box_ObjectDoor_Push": (41.9162, 387.2439),
    "Kinova3_Hollowbox_None_Shelf": (7.7897, 404.0659),
    "Panda_Plate_None_Push": (23.2818, 385.475),
    "Kinova3_Hollowbox_None_Trashcan": (12.2294, 415.3324),
    "Panda_Hollowbox_ObjectDoor_Trashcan": (13.6362, 308.0545),
    "Panda_Dumbbell_GoalWall_Push": (19.7563, 265.7354),
    "Panda_Dumbbell_ObjectDoor_Shelf": (26.4273, 283.3508),
    "IIWA_Dumbbell_ObjectDoor_Trashcan": (32.2144, 358.2904),
    "Panda_Hollowbox_None_PickPlace": (10.6354, 397.2718),
    "Panda_Plate_ObjectDoor_Shelf": (8.1509, 312.9323),
    "Panda_Plate_ObjectWall_Push": (11.0653, 321.3536),
}


MIMICGEN_ENVS = {
    # mimicgen --> min/max in dataset
    'CoffeePreparation_D0': (0.0, 1.0),
    'CoffeePreparation_D1': (0.0, 1.0),
    'Coffee_D0': (0.0, 1.0),
    'Coffee_D1': (0.0, 1.0),
    'Coffee_D2': (0.0, 1.0),
    'HammerCleanup_D0': (0.0, 1.0),
    'HammerCleanup_D1': (0.0, 1.0),
    'Kitchen_D0': (0.0, 1.0),
    'Kitchen_D1': (0.0, 1.0),
    'MugCleanup_D0': (0.0, 1.0),
    'MugCleanup_D1': (0.0, 1.0),
    'NutAssembly_D0': (0.0, 1.0),
    'PickPlace_D0': (0.0, 1.0),
    'Square_D0': (0.0, 1.0),
    'Square_D1': (0.0, 1.0),
    'Square_D2': (0.0, 1.0),
    'StackThree_D0': (0.0, 1.0),
    'StackThree_D1': (0.0, 1.0),
    'Stack_D0': (0.0, 1.0),
    'Stack_D1': (0.0, 1.0),
    'Threading_D0': (0.0, 1.0),
    'Threading_D1': (0.0, 1.0),
    'Threading_D2': (0.0, 1.0),
    'ThreePieceAssembly_D0': (0.0, 1.0),
    'ThreePieceAssembly_D1': (0.0, 1.0),
    'ThreePieceAssembly_D2': (0.0, 1.0),
}

for key in list(MIMICGEN_ENVS.keys()):
    for robot in ["Panda", "Sawyer"]:
        MIMICGEN_ENVS[f"{robot}_{key}"] = MIMICGEN_ENVS[key]

ENVID_TO_DNS = {**ENVID_TO_DNS, **MIMICGEN_ENVS}

# add scores for actual env ids
keys = list(ENVID_TO_DNS.keys())
for k in keys: 
    if k not in ATARI_NAME_TO_ENVID:
        continue
    envid = ATARI_NAME_TO_ENVID[k]
    ENVID_TO_DNS[envid] = ENVID_TO_DNS[k]


def get_data_normalized_score(task: str, raw_score: float, random_col=0, data_col=1) -> float:
    """Converts task score to data-normalized score."""
    scores = ENVID_TO_DNS.get(task, (math.nan, math.nan))
    random, data = scores[random_col], scores[data_col]
    return (raw_score - random) / (data - random)


def compute_random_dmcontrol_scores(): 
    random_scores = {}
    for envid in DM_CONTROL_ENVS:
        domain_name, task_name = envid.split("-")
        print(f"Computing random scores for {envid} ...")
        env = dmc2gym.make(domain_name=domain_name, task_name=task_name)
        random_scores[envid] = evaluate_random_policy(env)
    return random_scores


def evaluate_random_policy(env, n_eval_episodes=10):
    returns = []
    for _ in range(n_eval_episodes):
        _ = env.reset()
        done = False
        episode_return = 0
        while not done:
            action = env.action_space.sample()
            _, reward, done, _ = env.step(action)
            episode_return += reward
        returns.append(episode_return)
    return np.mean(returns)
