import json
from enum import Enum

##### UTILS #####


class EnumUtilities(Enum):
    """
    Mixin that allows to list the values of the enum
    """

    @classmethod
    def list_values(cls):
        return [member.value for member in cls]


class EnumEncoder(json.JSONEncoder):
    """
    Serialize enums manually because they aren't
    supported by json
    """

    def default(self, obj):
        if isinstance(obj, Enum):
            return obj.value
        return super().default(obj)


##### ENUMS #####


class FunctionSampling(EnumUtilities):
    SAMPLE_REJECTION = "sample-rejection"
    ENUMERATE = "enumerate"
    RANDOM = "random"


class NoiseMode(EnumUtilities):
    ADDITIVE = "additive"
    MULTIPLICATIVE = "multiplicative"
    FUNCTIONAL = "functional"


class MechanismFamily(EnumUtilities):
    LINEAR = "linear"
    NEURAL_NETWORK = "nn"
    TABULAR = "tabular"
    CUSTOM = "custom"  # provide your own python function


class NeuralNetworkType(EnumUtilities):
    # TODO: only FF is supported now
    FEEDFORWARD = "FF"
    CNN = "CNN"
    RNN = "RNN"
    TRANSFORMER = "transformer"


class NoiseDistribution(EnumUtilities):
    UNIFORM = "uniform"
    GAUSSIAN = "gaussian"
    GAUSSIAN_MIXTURE = "gaussian-mixture"


class VariableDataType(EnumUtilities):
    CONTINUOUS = "continuous"
    DISCRETE = "discrete"


class QueryType(EnumUtilities):
    # Level 1
    CONDITIONAL = "conditional"
    # Level 2
    CATE = "CATE"
    ATE = "ATE"
    ITE = "ITE"
    DTE = "DTE"
    CDTE = "CDTE"
    OIP = "outcome_interventional_probability"
    # Level 3
    CTF_DE = "Ctf-DE"
    CTF_IE = "Ctf-IE"
    CTF_TE = "Ctf-TE"


class VariableRole(EnumUtilities):
    # TODO: if not used remove, set it when intervened for example
    UNKNOWN = "unknown"
    OBSERVED = "observed"
    INTERVENED = "intervened"
    CONDITIONED = "conditioned"


class ErrorMetric(EnumUtilities):
    L1 = "L1"
    L2 = "L2"
    HAMMING = "hamming"
    COSINE = "cosine"
    MSE = "MSE"
    MAPE = "MAPE"


class KernelType(EnumUtilities):
    GAUSSIAN = "gaussian"
    UNIFORM = "uniform"
    EPANECHNIKOV = "epanechnikov"
    TRIANGULAR = "triangular"
    EPSILON = "epsilon"
    CUSTOM = "custom"


class IdentifiabilityRequirement(EnumUtilities):
    """
    Identifiability requirements in causal query sampling.

    - Level 1 queries are always considered identifiable
    - Level 2 & 3 queries use the y0 package (ID, IDC, ID*, IDC* algorithms)

    IMPORTANT LIMITATIONS:
    - Only works with Markovian graphs (no hidden confounders)
    """

    FORCE_IDENTIFIABLE = "force_identifiable"
    FORCE_NON_IDENTIFIABLE = "force_non_identifiable"
    NO_REQUIREMENT = "no_requirement"


class EstimabilityRequirement(EnumUtilities):
    """
    Estimability requirements in causal query sampling.
    Note that estimability is ONLY checked after identification,
    and it's currently only meaningful to check estimability if we also check identification
    (uses identified estimands and the observational dataset to check estimability)

    - For counterfactual estimability, check:
        - Identifiability (can we express the counterfactual in terms of observational data?)
        - Factual support: P(factual event) > 0
        - Conditional support: For all conditioning sets C in the identified estimand, P(C) > 0
    - For interventional estimability, check: Identifiability + Conditional support
    - For associational (e.g., conditional) estimability, check: Conditional support

    IMPORTANT LIMITATIONS:
    - Only provides meaningful estimability checking for DISCRETE variables
        - Continuous variables receive minimal checking (presence of data only, needs kernel density estimation to be realistic)
    - Only checked after identification, so if this is enabled, IdentifiabilityRequirement should not be equal to FORCE_NON_IDENTIFIABLE.
    - Basic estimability checking only:
        - verifies data support for conditioning events (context positivity)
        - for level 3 (e.g., CTF_TE): whether factual events have positive probability
        - does not check treatment overlap, covariate balance, sample size adequacy, or finite-sample statistical properties
    """

    FORCE_ESTIMABLE = "force_estimable"
    NO_REQUIREMENT = "no_requirement"
