#!/usr/bin/env python3



from hydra.utils import instantiate

from partnr.tools import PerceptionTool


class Agent:
    """
    This class represents an agent.
    Agent has access to various tools.
    """

    def __init__(self, uid, agent_conf, env_interface=None):
        # Assign the unique ID
        self.uid = uid

        self.agent_conf = agent_conf
        self.env_interface = env_interface
        self._dry_run = False

        # Initialize tools
        self.tools = self.__init_tools()

        # Log last used tool
        self.last_used_tool = None

    # Hashing Operator
    def __hash__(self):
        return hash(self.uid)

    # Equality operator
    def __eq__(self, other):
        #  Make sure that the other is of type Tool
        if not isinstance(other, Agent):
            return False

        # Return the equivalence based on the uid
        return self.uid == other.uid

    def reset(self):
        """Resets state variables of the agent"""

        # Reset state variables
        self.last_used_tool = None
        self._dry_run = False

        # Reset skills owned by this agent
        for tool_name in self.tools:
            if hasattr(self.tools[tool_name], "skill"):
                # Reset the skill's internal variables such as hidden states
                # and flag for tracking when the skill is entered for the first time
                self.tools[tool_name].skill.reset([0])
                # print(f"Reset {tool_name} skill for agent {self.uid}...")

    @property
    def agent_description(self):
        """Returns a string listing the agent's descriptions"""

        out = ""
        out += f"Agent ID: {self.uid}\n"
        for tool_name in sorted(self.tools.keys()):
            out += f"- {tool_name}: {self.tools[tool_name].description}\n"

        out += "\n"

        return out

    @property
    def tool_list(self):
        """Returns a string listing the agent's tools"""
        tool_names = sorted(self.tools.keys())
        return str(tool_names)

    @property
    def tool_descriptions(self):
        """Returns a string listing the agent's tools with its descriptions"""
        tool_names = sorted(self.tools.keys())
        return "\n".join(
            [
                f"- {tool_name}: {self.tools[tool_name].description}"
                for tool_name in tool_names
            ]
        )

    def __init_tools(self):
        # Declare as set to ensure uniqueness
        tools = {}

        for tool_category in self.agent_conf.tools:
            for tool_name in self.agent_conf.tools[tool_category]:
                print(f'processing tool: "{tool_name}"')
                tool_config = self.agent_conf.tools[tool_category][tool_name]
                tool = instantiate(tool_config)
                tool.agent_uid = self.uid

                # Motor skills require access to the environment and the device
                if "motor_skills" in tool_category:
                    tool.set_environment(self.env_interface)
                    tool.to(self.env_interface.device)

                # Perception requires access to the environment
                if "perception" in tool_category:
                    tool.set_environment(self.env_interface)

                # Make sure tool is not already added in the set
                if tool in tools:
                    print(f'tools already contains an instance of "{tool_name}"')
                    raise ValueError(
                        f'tools already contains an instance of "{tool_name}"'
                    )

                # Add tool to the set
                tools[tool.name] = tool

        return tools

    def pass_llm_to_tools(self, llm):
        """
        This method passes a given instance of LLM into the agent tools.
        Some tools require LLMs for their operation. However, its
        very expensive to maintain copies of LLMs in the memory
        so we need to share the instance of planner LLM across the tools.
        """
        for tool in self.tools.values():
            if hasattr(tool, "llm"):
                tool.set_llm(llm)

        return

    def get_tool_from_name(self, tool_name):
        if tool_name in self.tools:
            return self.tools[tool_name]
        raise ValueError(f'Tool "{tool_name}" not found')

    def get_last_state_description(self):
        # Make sure that the last used tool is not None
        if self.last_used_tool == None:
            return "Idle"

        return self.last_used_tool.get_state_description()

    def process_high_level_action(self, action, action_input, observations):
        """
        This method will consume high level actions to generate
        either a text response or a low level action.
        For every empty text response, this method should return a non empty low-level action.
        """
        # Fetch tool corresponding to the action
        try:
            tool = self.get_tool_from_name(action)
        except ValueError:
            return None, f'Tool "{action}" not found'

        # Process the high level action
        if self._dry_run and not isinstance(tool, PerceptionTool):
            return None, {f"agent_{self.uid}": f"{action} was a success"}
        low_level_action, response = tool.process_high_level_action(
            action_input, observations
        )

        # Set last used tool
        self.last_used_tool = tool

        return low_level_action, response
