#!/usr/bin/env python3



import os

import google.generativeai as genai
from google.generativeai.types import HarmBlockThreshold, HarmCategory


class Gemini:
    def __init__(self, conf):
        self.llm_conf = conf
        self.generation_params = self.llm_conf.generation_params
        self._validate_conf()
        self.verbose = self.llm_conf.verbose
        self.verbose = True
        self.model = genai.GenerativeModel(self.llm_conf.generation_params.engine)

    def _validate_conf(self):
        try:
            GOOGLE_API_KEY = os.getenv("GOOGLE_API_KEY")
            genai.configure(api_key=GOOGLE_API_KEY)
        except Exception:
            raise ValueError("No API keys provided")

    # @retry((Timeout, RateLimitError), tries=5, delay=60, backoff=2)
    def generate(self, prompt, stop=None, max_length=None):
        if stop is None:
            stop = self.generation_params.stop

        if max_length is None:
            max_length = self.generation_params.max_tokens

        # Setup the generation parameters.
        generation_config = genai.types.GenerationConfig(
            candidate_count=1,
            stop_sequences=[stop],
            max_output_tokens=max_length,
            temperature=self.generation_params.temperature,
        )

        # Safety settings
        # Available categories: HATE_SPEECH, HARASSMENT, SEXUALLY_EXPLICIT, DANGEROUS_CONTENT
        # Rating Options: BLOCK_NONE, BLOCK_ONLY_HIGH, BLOCK_MEDIUM_AND_ABOVE (default), BLOCK_LOW_AND_ABOVE
        # The default settings were found to block planning responses in some cases.
        safety_settings = {
            HarmCategory.HARM_CATEGORY_HATE_SPEECH: HarmBlockThreshold.BLOCK_MEDIUM_AND_ABOVE,
            HarmCategory.HARM_CATEGORY_HARASSMENT: HarmBlockThreshold.BLOCK_MEDIUM_AND_ABOVE,
            HarmCategory.HARM_CATEGORY_SEXUALLY_EXPLICIT: HarmBlockThreshold.BLOCK_MEDIUM_AND_ABOVE,
            HarmCategory.HARM_CATEGORY_DANGEROUS_CONTENT: HarmBlockThreshold.BLOCK_MEDIUM_AND_ABOVE,
        }

        # Generate the response.
        self.response = self.model.generate_content(
            prompt,
            generation_config=generation_config,
            safety_settings=safety_settings,
        )

        # Return explaination of the response is blocked due to safety reasons.
        try:
            return self.response.text
        except ValueError:
            return "Response was blocked for safety reason. Please tweak your query to generate different response."
