#!/usr/bin/env python3



import regex as re

from partnr.agent.env.scene.scene_parser import get_pretty_scene_string
from partnr.llm import instantiate_llm

prompt = """
There's a house with the following objects and furniture:

{initial_env_state}

Provide {k} instructions to rearrange objects in the house, following the next principles:
1. Each instruction should be fulfilled by Navigation, Picking up and Placing objects.
2. The instructions should different from each other in terms of difficulty and complexity.
    2.1. Some instructions should be multi-step (moving several objects in one instruction) and some should be single-step (only one object is specified and needs to be rearranged).
    2.2. In some instructions the initial position of the object should be specified, in others should be inferred.
3. The instructions should be abstract and semantically rich, in particular they should refer to classes or groups of objects.
4. The instructions should be meaningful.
5. Don't use objects or receptacles that are not listed on top!
6. The instructions shouldn't be detailed and explain the step by step, but more high-level.

Follow the next format:
1. ...
2. ...
...

Go:
"""  # noqa


class InstructionGenerator:
    def __init__(self, scene):
        """
        scene: SceneParser
        """
        self.scene = scene
        self.llm = instantiate_llm("openai_chat", generation_params={"model": "gpt-4"})
        self.scene_str, self.receptacle, self.objects = get_pretty_scene_string(
            scene.grouped_objects, remove_underscores_and_numbers=True
        )

    def generate_instructions(self, k=5, max_length=1000):
        formatted_prompt = prompt.format(initial_env_state=self.scene_str, k=k)
        llm_answer = self.llm.generate(
            formatted_prompt,
            max_length=max_length,
            keep_message_history=False,
            request_timeout=60,
            temperature=1.0,  # default is 0, we want higher temperature to get more diverse instructions
        )
        self.instructions = self.parse_instructions(llm_answer)
        return self.instructions

    def parse_instructions(self, llm_answer):
        return re.split(r"\d+\. ", llm_answer)[1:]
