# ModelTransformer

[```[Documentation]```](docs/Readme.md)

A Python library for calibrating neural network outputs using various methods.

## Installation

```
pip install model-transformer
```

## Available Methods

ModelTransformer provides the following calibration methods:

| Method | Description | Paper |
|--------|-------------|-------|
| Temperature Scaling | Simple and effective method that learns a single parameter to scale the logits. | [On Calibration of Modern Neural Networks](https://arxiv.org/abs/1706.04599) |
| Ensemble Temperature Scaling | An extension of temperature scaling that combines multiple temperatures. | [ Mix-n-Match: Ensemble and Compositional Methods for Uncertainty Calibration in Deep Learning](http://arxiv.org/abs/2003.07329) |
| Isotonic Regression Calibration | Uses isotonic regression to calibrate probabilities. | [Mix-n-Match: Ensemble and Compositional Methods for Uncertainty Calibration in Deep Learning](https://www.cs.cornell.edu/~alexn/papers/calibration.icml05.crc.rev3.pdf) |
| Spline Calibration | Applies spline interpolation for calibration. | [Calibration of Neural Networks using Splines](https://arxiv.org/abs/2006.12800) |

Each method follows a scikit-learn-like API with `fit`, `transform`, and `fit_transform` methods.

## Usage

```python
import numpy as np
from model_transformer import TemperatureScaling, EnsembleTemperatureScaling, IsotonicRegressionCalibration, SplineCalibration, one_hot_encode

# Generate sample data
logits = np.random.randn(1000, 10)
true_labels = np.random.randint(0, 10, 1000)
one_hot_labels = one_hot_encode(true_labels, 10)
eval_logits = np.random.randn(100, 10)

# Temperature Scaling
ts = TemperatureScaling()
ts.fit(logits, one_hot_labels)
calibrated_probs = ts.transform(eval_logits)
print(calibrated_probs.shape)  # (100, 10)

# Ensemble Temperature Scaling
ets = EnsembleTemperatureScaling()
ets.fit(logits, one_hot_labels)
calibrated_probs = ets.transform(eval_logits)
print(calibrated_probs.shape)  # (100, 10)

# Isotonic Regression Calibration
irc = IsotonicRegressionCalibration()
irc.fit(logits, one_hot_labels)
calibrated_probs = irc.transform(eval_logits)
print(calibrated_probs.shape)  # (100, 10)

# Spline Calibration
sc = SplineCalibration(spline_method='natural', splines=10)
sc.fit(logits, one_hot_labels)
calibrated_probs = sc.transform(eval_logits)
print(calibrated_probs.shape)  # (100, 10)
```

## Method Details

### TemperatureScaling

```python
ts = TemperatureScaling(loss='ce')
```

- `loss`: Loss function to use for optimization ('ce', 'mse', or 'brier').

### EnsembleTemperatureScaling

```python
ets = EnsembleTemperatureScaling(loss='ce')
```

- `loss`: Loss function to use for optimization ('ce', 'mse', or 'brier').

### IsotonicRegressionCalibration

```python
irc = IsotonicRegressionCalibration()
```

### SplineCalibration

```python
sc = SplineCalibration(spline_method='natural', splines=6)
```

- `spline_method`: The spline method to use ('natural', 'parabolic', or 'cubic').
- `splines`: The number of spline knots to use.

## Utility Functions

- `one_hot_encode(labels, num_classes)`: Converts integer labels to one-hot encoded format.