import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid

from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, get_model_name_from_path, KeywordsStoppingCriteria

from PIL import Image
import math

from AttnAdapter import AttnAdapter
from add_noise import add_diffusion_noise
from cd_sample import evolve_cd_greedy_search

import matplotlib.pyplot as plt
from matplotlib.colors import ListedColormap
import numpy as np

evolve_cd_greedy_search()

def split_list(lst, n):
    """Split a list into n (roughly) equal-sized chunks"""
    chunk_size = math.ceil(len(lst) / n)  # integer division
    return [lst[i:i+chunk_size] for i in range(0, len(lst), chunk_size)]

def get_chunk(lst, n, k):
    chunks = split_list(lst, n)
    return chunks[k]


import os
import torch
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.colors import ListedColormap

# def visualize_and_save_heads_plogp(attentions,
#                                    out_dir,
#                                    img_start_idx=35,
#                                    img_size=(24, 24),
#                                    layer_idx=14):
#     """
#     对第 layer_idx 层的每个头计算 p·log(p) 并保存热图：
#       - 每个头一张 24×24 的二值图，前10%位置标注为蓝色，其余为白色
#     """
#     os.makedirs(out_dir, exist_ok=True)

#     # 提取注意力并切到 CPU
#     attn_layer = attentions[layer_idx]               # (1, heads, Q, K)
#     _, num_heads, _, _ = attn_layer.shape
#     H, W = img_size
#     L = H * W

#     # 切出最后一个 query 的视觉 token 部分，并转 float32
#     head_p = (
#         attn_layer[0, :, -1, img_start_idx: img_start_idx + L]
#         .detach()
#         .cpu()
#         .float()
#         .clamp(min=1e-12)
#     )  # shape (heads, L)

#     # 计算 p·log(p)
#     head_plogp = - head_p * torch.log(head_p)         # (heads, L)
#     maps = head_plogp.reshape(num_heads, H, W)       # (heads, H, W)

#     # 二值 colormap：0->白, 1->蓝
#     cmap = ListedColormap(['white', '#5190d2'])

#     # 保存每个头的二值热图
#     for i in range(num_heads):
#         arr = maps[i].numpy()
#         # thresh = np.quantile(arr.flatten(), 0.95)   # 前10%阈值
#         # mask = (arr >= thresh).astype(int)          # 二值化

#         thresh = np.quantile(arr.flatten(), 0.05)   # 最低5%阈值
#         mask = (arr <= thresh).astype(int)          # 二值化: <=阈值为1(蓝)，其余为0(白)

#         fig, ax = plt.subplots(figsize=(3, 3))
#         ax.imshow(mask, cmap=cmap, interpolation='nearest')
#         ax.axis('off')

#         svg_path = os.path.join(out_dir, f"head_{i:02d}_top10.svg")
#         fig.savefig(svg_path, format='svg', bbox_inches='tight', pad_inches=0)
#         plt.close(fig)

#     print(f"Saved {num_heads} head masks (top 10%) as SVGs in '{out_dir}/'")

def visualize_and_save_heads_plogp(attentions,
                                   out_dir,           # 保存目录
                                   img_start_idx=35,
                                   img_size=(24, 24),
                                   layer_idx=14):
    """
    对第 layer_idx 层的每个头计算 p·log(p) 并保存热图：
      - 每个头一张 24×24 图，文件名 head_00.png ... head_31.png
      - 一张合并图 all_heads.png
    """
    os.makedirs(out_dir, exist_ok=True)

    # 提取注意力并切到 CPU
    attn_layer = attentions[layer_idx]               # (1, heads, Q, K)
    _, num_heads, _, _ = attn_layer.shape
    H, W = img_size
    L = H * W

    # 拷贝到 CPU，并取最后一个 query 的视觉 token 部分
    head_p = (
        attn_layer[0, :, -1, img_start_idx : img_start_idx + L]
        .detach()
        .cpu()
        .float()                 # <-- 转为 float32
        .clamp(min=1e-12)        # 现在 clamp 可以正常工作
    )  # shape (heads, L)

    # 计算 p·log(p)，再 reshape
    head_plogp = - head_p * torch.log(head_p)          # (heads, L)
    # head_plogp = head_p
    maps = head_plogp.reshape(num_heads, H, W)       # (heads, H, W)

    # 单独保存每个头
    for i in range(num_heads):
        img_np = maps[i].numpy()                     # CPU 上的 numpy 数组
        plt.figure(figsize=(3,3))
        plt.imshow(img_np, cmap='Blues', interpolation='nearest')
        plt.axis('off')
        plt.savefig(os.path.join(out_dir, f"cp_head_{i:02d}.svg"),
                    bbox_inches='tight', pad_inches=0)
        plt.close()

    # 保存合并大图
    fig, axes = plt.subplots(4, 8, figsize=(16, 8))
    for i, ax in enumerate(axes.flat):
        img_np = maps[i].numpy()  # 转 numpy
        ax.imshow(img_np, cmap='Blues', interpolation='nearest')
        ax.axis('off')
    plt.tight_layout(pad=0.1)
    fig.suptitle(f'Layer {layer_idx+1} p·log(p) per Head', fontsize=12)
    merged_path = os.path.join(out_dir, "all_heads_fh.png")
    plt.savefig(merged_path, bbox_inches='tight', pad_inches=0.1)
    plt.close()

    print(f"Saved {num_heads} head maps + combined figure to '{out_dir}/'")




def eval_model(args):
    # Model
    disable_torch_init()
    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)
    tokenizer, model, image_processor, context_len = load_pretrained_model(model_path, args.model_base, model_name)

    if args.use_attn_aug == True:
        for i, layer in enumerate(model.model.layers):
            if i > 10 and i < 15:
                attn_adap = AttnAdapter(layer.self_attn.config)
                attn_adap.load_state_dict(layer.self_attn.state_dict())
                attn_adap = attn_adap.half().cuda()
                layer.self_attn = attn_adap

    questions = [json.loads(q) for q in open(os.path.expanduser(args.question_file), "r")]
    questions = get_chunk(questions, args.num_chunks, args.chunk_idx)
    # answers_file = os.path.expanduser(args.answers_file)
    # os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    # ans_file = open(answers_file, "w")
    for line in tqdm(questions):
        idx = line["question_id"]
        if idx != 13:
            continue
        image_file = line["image"]
        qs = line["text"]
        label = line["label"]
        cur_prompt = qs
        if model.config.mm_use_im_start_end:
            qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
        else:
            qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

        qs = qs + "\nAnswer the question using a single word or phrase."

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        input_ids = tokenizer_image_token(prompt, tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt').unsqueeze(0).cuda()

        # cpu_ids = input_ids[0].cpu().tolist()
        # tokens = []
        # for idx in cpu_ids:
        #     if idx == IMAGE_TOKEN_INDEX:
        #         tokens.append(DEFAULT_IMAGE_TOKEN)
        #     else:
        #         try:
        #            tokens.append(tokenizer.convert_ids_to_tokens(idx))
        #         except IndexError:
        #             tokens.append(f"<UNK_ID_{idx}>")
 
        # print("=== input tokens projection ===")
        # for pos, (tid, tok) in enumerate(zip(cpu_ids, tokens)):
        #     print(f"  pos {pos:03d}: id {tid:>5d} → token `{tok}`")
        # print("=========================")

        image = Image.open(os.path.join(args.image_folder, image_file))
        image_tensor = image_processor.preprocess(image, return_tensors='pt')['pixel_values'][0]

        if args.use_cd:
            image_tensor_cd = add_diffusion_noise(image_tensor, args.noise_step)
        else:
            image_tensor_cd = None 

        stop_str = conv.sep if conv.sep_style != SeparatorStyle.TWO else conv.sep2
        keywords = [stop_str]
        stopping_criteria = KeywordsStoppingCriteria(keywords, tokenizer, input_ids)

        with torch.inference_mode():
            output_ids = model.generate(
                input_ids,
                images=image_tensor.unsqueeze(0).half().cuda(),
                images_cd=(image_tensor_cd.unsqueeze(0).half().cuda() if image_tensor_cd is not None else None),
                do_sample=True if args.temperature > 0 else False,
                temperature=args.temperature,
                top_p=args.top_p,
                num_beams=args.num_beams,
                max_new_tokens=1024,
                use_cache=True,
                output_scores=True,
                output_attentions=True,
                return_dict_in_generate=True
                )

        input_token_len = input_ids.shape[1]
        n_diff_input_output = (input_ids != output_ids.sequences[:, :input_token_len]).sum().item()
        if n_diff_input_output > 0:
            print(f'[Warning] {n_diff_input_output} output_ids are not the same as the input_ids')
        outputs = tokenizer.batch_decode(output_ids.sequences[:, input_token_len:], skip_special_tokens=True)[0]
        outputs = outputs.strip()
        if outputs.endswith(stop_str):
            outputs = outputs[:-len(stop_str)]
        outputs = outputs.strip()

        token_id = output_ids.sequences[:, input_token_len].item()
        probs = torch.softmax(output_ids.scores[0], dim=-1)[0, token_id].item()  

        # for l_idx, layer in enumerate(model.model.layers):
        #     if l_idx == 14:
        #         entropy = layer.self_attn.spatial_entropy.detach().cpu()

        print("probs:", probs)
        print("keys:", output_ids.keys())

        # attns = output_ids.attentions
        # print(len(attns))
        # print(len(attns[0]))
        # print(attns[0][0].shape)
        # print(attns[1][0].shape)

        attns = output_ids.attentions[0]
        # print(os.path.join(args.attn_output_dir, f"attn_{idx}.pt"))
        visualize_and_save_heads_plogp(attns, out_dir=args.attn_output_dir)
        # torch.save(attns, os.path.join(args.attn_output_dir, f"attn_{idx}.pt"))

    #     ans_id = shortuuid.uuid()
    #     ans_file.write(json.dumps({"question_id": idx,
    #                                "text": cur_prompt,
    #                                "label": label,
    #                                "pred": outputs,
    #                                "probs": probs,
    #                                "entropy": entropy.item(),
    #                                }) + "\n")
    #     ans_file.flush()
    # ans_file.close()

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model-path", type=str, default="facebook/opt-350m")
    parser.add_argument("--model-base", type=str, default=None)
    parser.add_argument("--image-folder", type=str, default="")
    parser.add_argument("--question-file", type=str, default="tables/question.jsonl")
    parser.add_argument("--answers-file", type=str, default="answer.jsonl")
    parser.add_argument("--conv-mode", type=str, default="llava_v1")
    parser.add_argument("--num-chunks", type=int, default=1)
    parser.add_argument("--chunk-idx", type=int, default=0)
    parser.add_argument("--temperature", type=float, default=0.2)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=1)
    parser.add_argument("--use-attn-aug", action='store_true', default=False)   
    parser.add_argument("--use-cd", action='store_true', default=False) 
    parser.add_argument("--noise-step", type=int, default=999) 
    parser.add_argument("--attn-output-dir", type=str, default="attn_outputs")     
    args = parser.parse_args()

    eval_model(args)