"""
I am not sure what kMeans implementation we should use,
so let's create an interface for it,
to be able to switch between different implementations easily.
"""

from typing import Protocol

import numpy as np


class ClusteringDiscretizer(Protocol):
    def __init__(
        self,
        n_clusters: int = 8,
        max_iter: int = 300,
        tol: float = 1e-4,
        use_gpu: bool = False,
        **kwargs
    ) -> None:
        """
        Initialize the underlying clustering model.
        Common parameters are provided as defaults, and additional parameters can be passed as kwargs.

        Args:
            n_clusters (int): Number of clusters.
            max_iter (int): Maximum number of iterations.
            tol (float): Tolerance to declare convergence.
            use_gpu (bool): Whether to use GPU for acceleration (implementation-dependent).
            kwargs: Additional parameters specific to the implementation.
        """
        ...

    def train(self, data: np.ndarray) -> None:
        """
        Train the clustering model using the provided data.
        Args:
            data (np.ndarray): A 2D numpy array with shape (n_samples, n_features).
        """
        ...

    def discrete(self, vectors: np.ndarray) -> np.ndarray[int]:
        """
        Assign the input vectors to clusters and return the corresponding cluster indices.
        Args:
            vectors (list[np.ndarray]): A list of vectors to be clustered.

        Returns:
            np.ndarray[int]: An array of integers representing the cluster assignments
        """
        ...

    def save(self, path: str) -> None:
        """
        Save the trained model to the specified path.
        Args:
            path (str): The file path to save the model.
        """
        ...

    @classmethod
    def load(cls, path: str) -> "ClusteringDiscretizer":
        """
        Load a pre-trained model from the specified path and return an instance of ClusteringDiscretizer.
        Args:
            path (str): The file path from which to load the model.

        Returns:
            ClusteringDiscretizer: An instance of the class implementing this interface.
        """
        ...
